/***********************************************************************
*
* Copyright (c) 2012-2026 Barbara Geller
* Copyright (c) 2012-2026 Ansel Sermersheim
*
* Copyright (c) 2015 The Qt Company Ltd.
* Copyright (c) 2012-2016 Digia Plc and/or its subsidiary(-ies).
* Copyright (c) 2008-2012 Nokia Corporation and/or its subsidiary(-ies).
*
* This file is part of CopperSpice.
*
* CopperSpice is free software. You can redistribute it and/or
* modify it under the terms of the GNU Lesser General Public License
* version 2.1 as published by the Free Software Foundation.
*
* CopperSpice is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
*
* https://www.gnu.org/licenses/
*
***********************************************************************/

#include <qdockarealayout_p.h>
#include <qmainwindowlayout_p.h>

#ifndef QT_NO_MAINWINDOW

#include <qapplication.h>
#include <qdebug.h>
#include <qdockwidget.h>
#include <qmainwindow.h>
#include <qmap.h>
#include <qrubberband.h>
#include <qstack.h>
#include <qstatusbar.h>
#include <qstring.h>
#include <qstyle.h>
#include <qstylepainter.h>
#include <qtextstream.h>
#include <qtimer.h>
#include <qtoolbar.h>
#include <qvarlengtharray.h>

#include <qapplication_p.h>
#include <qdockwidget_p.h>
#include <qlayoutengine_p.h>
#include <qtabbar_p.h>
#include <qtoolbar_p.h>
#include <qtoolbarlayout_p.h>
#include <qwidgetanimator_p.h>
#include <qwidgetresizehandler_p.h>

extern QMainWindowLayout *qt_mainwindow_layout(const QMainWindow *window);

#if ! defined(QT_NO_DOCKWIDGET)

static void dumpLayout(QTextStream &qout, const QDockAreaLayoutInfo &layout, QString indent);

static void dumpLayout(QTextStream &qout, const QDockAreaLayoutItem &item, QString indent)
{
   qout << indent << "QDockAreaLayoutItem: "
        << "pos: " << item.pos << " size:" << item.size
        << " gap:" << (item.flags & QDockAreaLayoutItem::GapItem)
        << " keepSize:" << (item.flags & QDockAreaLayoutItem::KeepSize) << '\n';

   indent += "  ";

   if (item.widgetItem != nullptr) {
      qout << indent << "widget: "
         << item.widgetItem->widget()->metaObject()->className()
         << " \"" << item.widgetItem->widget()->windowTitle() << "\"\n";

   } else if (item.subinfo != nullptr) {
      qout << indent << "subinfo:\n";
      dumpLayout(qout, *item.subinfo, indent + "  ");

   } else if (item.placeHolderItem != nullptr) {
      QRect r = item.placeHolderItem->topLevelRect;

      qout << indent << "placeHolder: "
         << "pos: " << item.pos << " size:" << item.size
         << " gap:" << (item.flags & QDockAreaLayoutItem::GapItem)
         << " keepSize:" << (item.flags & QDockAreaLayoutItem::KeepSize)
         << " objectName:" << item.placeHolderItem->objectName
         << " hidden:" << item.placeHolderItem->hidden
         << " window:" << item.placeHolderItem->window
         << " rect:" << r.x() << ',' << r.y() << ' '
         << r.width() << 'x' << r.height() << '\n';
   }
}

static void dumpLayout(QTextStream &qout, const QDockAreaLayoutInfo &layout, QString indent)
{
   const QSize minSize = layout.minimumSize();

   qout << indent << "QDockAreaLayoutInfo: "
      << layout.m_dockAreaInfoRect.left() << ','
      << layout.m_dockAreaInfoRect.top() << ' '
      << layout.m_dockAreaInfoRect.width() << 'x'
      << layout.m_dockAreaInfoRect.height()
      << " min size: " << minSize.width() << ',' << minSize.height()
      << " orient:"  << layout.m_dockAreaOrientation
      << " tabbed:"  << layout.tabbed
      << " tbshape:" << layout.tabBarShape << '\n';

   indent += "  ";

   for (int i = 0; i < layout.item_list.count(); ++i) {
      qout << indent << "Item: " << i << '\n';
      dumpLayout(qout, layout.item_list.at(i), indent + "  ");
   }

}

static void dumpLayout(QTextStream &qout, const QDockAreaLayout &layout)
{
   qout << "QDockAreaLayout: "
      << layout.m_dockAreaRect.left() << ','
      << layout.m_dockAreaRect.top() << ' '
      << layout.m_dockAreaRect.width() << 'x'
      << layout.m_dockAreaRect.height() << '\n';

   qout << "TopDockArea:\n";
   dumpLayout(qout, layout.m_docks[QInternal::TopDock],    "  ");

   qout << "LeftDockArea:\n";
   dumpLayout(qout, layout.m_docks[QInternal::LeftDock],   "  ");

   qout << "RightDockArea:\n";
   dumpLayout(qout, layout.m_docks[QInternal::RightDock],  "  ");

   qout << "BottomDockArea:\n";
   dumpLayout(qout, layout.m_docks[QInternal::BottomDock], "  ");
}

QDebug operator<<(QDebug debug, const QDockAreaLayout &layout)
{
   QString s;
   QTextStream str(&s);
   dumpLayout(str, layout);
   debug << s;

   return debug;
}

QDebug operator<<(QDebug debug, const QMainWindowLayout *layout)
{
   debug << layout->layoutState.dockAreaLayout;
   return debug;
}

class QDockWidgetGroupLayout : public QLayout
{
 public:
   QDockWidgetGroupLayout(QWidget *parent)
      : QLayout(parent)
   {
      setSizeConstraint(QLayout::SetMinAndMaxSize);
      resizer = new QWidgetResizeHandler(parent);
      resizer->setMovingEnabled(false);
   }

   ~QDockWidgetGroupLayout() {
      info.deleteAllLayoutItems();
   }

   void addItem(QLayoutItem *) override {
      // error, may want to throw?
   }

   int count() const override {
      return 0;
   }

   QLayoutItem *itemAt(int index) const override {
      int x = 0;
      return info.itemAt(&x, index);
   }

   QLayoutItem *takeAt(int index) override {
      int x = 0;
      return info.takeAt(&x, index);
   }

   QSize sizeHint() const override {
      int fw = frameWidth();
      return info.sizeHint() + QSize(fw, fw);
   }

   QSize minimumSize() const override {
      int fw = frameWidth();
      return info.minimumSize() + QSize(fw, fw);
   }

   QSize maximumSize() const override {
      int fw = frameWidth();
      return info.maximumSize() + QSize(fw, fw);
   }

   void setGeometry(const QRect &r) override {
      static_cast<QDockWidgetGroupWindow *>(parent())->destroyOrHideIfEmpty();

      QDockAreaLayoutInfo *li = layoutInfo();

      if (li->isEmpty()) {
         return;
      }

      int fw = frameWidth();
      li->reparentWidgets(parentWidget());
      li->m_dockAreaInfoRect = r.adjusted(fw, fw, -fw, -fw);
      li->fitItems();
      li->apply(false);
      resizer->setActive(QWidgetResizeHandler::Resize, !nativeWindowDeco());
   }

   QDockAreaLayoutInfo *layoutInfo() {
      return &info;
   }

   bool nativeWindowDeco() const {
      return QDockWidgetLayout::wmSupportsNativeWindowDeco();
   }

   int frameWidth() const {
      return nativeWindowDeco() ? 0 :
         parentWidget()->style()->pixelMetric(QStyle::PM_DockWidgetFrameWidth, nullptr, parentWidget());
   }

 private:
   QDockAreaLayoutInfo info;
   QWidgetResizeHandler *resizer;
};

bool QDockWidgetGroupWindow::event(QEvent *e)
{
   switch (e->type()) {
      case QEvent::Close:
         // Forward the close to the QDockWidget just as if its close button was pressed
         if (QDockWidget *dw = topDockWidget()) {
            e->ignore();
            dw->close();
            adjustFlags();
         }
         return true;

      case QEvent::Move:
         // Let QDockWidgetPrivate::moseEvent handle the dragging
         if (QDockWidget *dw = topDockWidget()) {
            static_cast<QDockWidgetPrivate *>(QWidgetPrivate::get(dw))->moveEvent(static_cast<QMoveEvent *>(e));
         }
         return true;

      case QEvent::NonClientAreaMouseMove:
      case QEvent::NonClientAreaMouseButtonPress:
      case QEvent::NonClientAreaMouseButtonRelease:
      case QEvent::NonClientAreaMouseButtonDblClick:
         // Let the QDockWidgetPrivate of the currently visible dock widget handle the drag and drop
         if (QDockWidget *dw = topDockWidget()) {
            static_cast<QDockWidgetPrivate *>(QWidgetPrivate::get(dw))->nonClientAreaMouseEvent(static_cast<QMouseEvent *>(e));
         }
         return true;

      case QEvent::ChildAdded:
         if (qobject_cast<QDockWidget *>(static_cast<QChildEvent *>(e)->child())) {
            adjustFlags();
         }
         break;

      case QEvent::LayoutRequest:
         // We might need to show the widget again
         destroyOrHideIfEmpty();
         break;
      default:
         break;
   }

   return QWidget::event(e);
}

void QDockWidgetGroupWindow::paintEvent(QPaintEvent *)
{
   QDockWidgetGroupLayout *lay = static_cast<QDockWidgetGroupLayout *>(layout());
   bool nativeDeco = lay->nativeWindowDeco();

   if (! nativeDeco) {
      QStyleOptionFrame framOpt;
      framOpt.initFrom(this);
      QStylePainter p(this);
      p.drawPrimitive(QStyle::PE_FrameDockWidget, framOpt);
   }
}

QDockAreaLayoutInfo *QDockWidgetGroupWindow::layoutInfo() const
{
   return static_cast<QDockWidgetGroupLayout *>(layout())->layoutInfo();
}

QDockWidget *QDockWidgetGroupWindow::topDockWidget() const
{
   QDockAreaLayoutInfo *info = layoutInfo();
   QDockWidget *dw = nullptr;

   if (info->tabBar && info->tabBar->currentIndex() >= 0) {
      int i = info->tabIndexToListIndex(info->tabBar->currentIndex());
      if (i >= 0) {
         const QDockAreaLayoutItem &item = info->item_list.at(i);
         if (item.widgetItem) {
            dw = qobject_cast<QDockWidget *>(item.widgetItem->widget());
         }
      }
   }

   if (! dw) {
      for (int i = 0; !dw && i < info->item_list.count(); ++i) {
         const QDockAreaLayoutItem &item = info->item_list.at(i);
         if (item.skip()) {
            continue;
         }
         if (!item.widgetItem) {
            continue;
         }
         dw = qobject_cast<QDockWidget *>(item.widgetItem->widget());
      }
   }
   return dw;
}

void QDockWidgetGroupWindow::destroyOrHideIfEmpty()
{
   if (!layoutInfo()->isEmpty()) {
      show(); // It might have been hidden,
      return;
   }
   // There might still be placeholders
   if (! layoutInfo()->item_list.isEmpty()) {
      hide();
      return;
   }

   // Make sure to reparent the possibly floating or hidden QDockWidgets to the parent
   for (QDockWidget *dw : findChildren<QDockWidget *>(QString(), Qt::FindDirectChildrenOnly)) {
      bool wasFloating = dw->isFloating();
      bool wasHidden   = dw->isHidden();
      dw->setParent(parentWidget());

      if (wasFloating) {
         dw->setFloating(true);
      } else {
         // maybe it was hidden, we still have to put it back in the main layout.
         QMainWindowLayout *ml = qt_mainwindow_layout(static_cast<QMainWindow *>(parentWidget()));
         Qt::DockWidgetArea area = ml->dockWidgetArea(this);

         if (area == Qt::NoDockWidgetArea) {
            area = Qt::LeftDockWidgetArea;
         }
         static_cast<QMainWindow *>(parentWidget())->addDockWidget(area, dw);
      }

      if (!wasHidden) {
         dw->show();
      }
   }

   for (QTabBar *tb : findChildren<QTabBar *>(QString(), Qt::FindDirectChildrenOnly)) {
      tb->setParent(parentWidget());
   }

   deleteLater();
}


void QDockWidgetGroupWindow::adjustFlags()
{
   QDockWidget *top = topDockWidget();
   if (!top) {
      return;
   }
   const bool nativeDeco = static_cast<QDockWidgetGroupLayout *>(layout())->nativeWindowDeco();
   Qt::WindowFlags oldFlags = windowFlags();
   Qt::WindowFlags flags = oldFlags;

   if (nativeDeco) {
      flags |= Qt::CustomizeWindowHint | Qt::WindowTitleHint;
      if (top->features() & QDockWidget::DockWidgetClosable) {
         flags |= Qt::WindowCloseButtonHint;
      } else {
         flags &= ~Qt::WindowCloseButtonHint;
      }
      flags &= ~Qt::FramelessWindowHint;
   } else {
      flags |= Qt::FramelessWindowHint;
   }
   if (oldFlags != flags) {
      setWindowFlags(flags);
      show(); // setWindowFlags hides the window
   }

   setWindowTitle(top->windowTitle());
   setWindowIcon(top->windowIcon());
}
#endif

// we deal with all the #ifndef code here so QMainWindowLayout code is clean

QMainWindowLayoutState::QMainWindowLayoutState(QMainWindow *win)
   :
#ifndef QT_NO_TOOLBAR
   toolBarAreaLayout(win),
#endif

#ifndef QT_NO_DOCKWIDGET
   dockAreaLayout(win)
#else
   centralWidgetItem(0)
#endif

{
   mainWindow = win;
}

QSize QMainWindowLayoutState::sizeHint() const
{
   QSize result(0, 0);

#ifndef QT_NO_DOCKWIDGET
   result = dockAreaLayout.sizeHint();
#else
   if (centralWidgetItem != nullptr) {
      result = centralWidgetItem->sizeHint();
   }
#endif

#ifndef QT_NO_TOOLBAR
   result = toolBarAreaLayout.sizeHint(result);
#endif

   return result;
}

QSize QMainWindowLayoutState::minimumSize() const
{
   QSize result(0, 0);

#ifndef QT_NO_DOCKWIDGET
   result = dockAreaLayout.minimumSize();
#else
   if (centralWidgetItem != nullptr) {
      result = centralWidgetItem->minimumSize();
   }
#endif

#ifndef QT_NO_TOOLBAR
   result = toolBarAreaLayout.minimumSize(result);
#endif

   return result;
}

void QMainWindowLayoutState::apply(bool animated)
{
#ifndef QT_NO_TOOLBAR
   toolBarAreaLayout.apply(animated);
#endif

#ifndef QT_NO_DOCKWIDGET
   //  dumpLayout(dockAreaLayout, QString());
   dockAreaLayout.apply(animated);
#else

   if (centralWidgetItem != nullptr) {
      QMainWindowLayout *layout = qt_mainwindow_layout(mainWindow);
      Q_ASSERT(layout != nullptr);

      layout->widgetAnimator.animate(centralWidgetItem->widget(), centralWidgetRect, animated);
   }
#endif
}

void QMainWindowLayoutState::fitLayout()
{
   QRect r;

#ifdef QT_NO_TOOLBAR
   r = rect;
#else
   toolBarAreaLayout.m_toolBarAreaRect = rect;
   r = toolBarAreaLayout.fitLayout();
#endif

#ifndef QT_NO_DOCKWIDGET
   dockAreaLayout.m_dockAreaRect = r;
   dockAreaLayout.fitLayout();
#else
   centralWidgetRect = r;
#endif
}

void QMainWindowLayoutState::deleteAllLayoutItems()
{
#ifndef QT_NO_TOOLBAR
   toolBarAreaLayout.deleteAllLayoutItems();
#endif

#ifndef QT_NO_DOCKWIDGET
   dockAreaLayout.deleteAllLayoutItems();
#endif
}

void QMainWindowLayoutState::deleteCentralWidgetItem()
{
#ifndef QT_NO_DOCKWIDGET
   delete dockAreaLayout.centralWidgetItem;
   dockAreaLayout.centralWidgetItem = nullptr;
#else
   delete centralWidgetItem;
   centralWidgetItem = nullptr;
#endif
}

QLayoutItem *QMainWindowLayoutState::itemAt(int index, int *x) const
{
#ifndef QT_NO_TOOLBAR
   if (QLayoutItem *ret = toolBarAreaLayout.itemAt(x, index)) {
      return ret;
   }
#endif

#ifndef QT_NO_DOCKWIDGET
   if (QLayoutItem *ret = dockAreaLayout.itemAt(x, index)) {
      return ret;
   }
#else
   if (centralWidgetItem != nullptr && (*x)++ == index) {
      return centralWidgetItem;
   }
#endif

   return nullptr;
}

QLayoutItem *QMainWindowLayoutState::takeAt(int index, int *x)
{
#ifndef QT_NO_TOOLBAR
   if (QLayoutItem *ret = toolBarAreaLayout.takeAt(x, index)) {
      return ret;
   }
#endif

#ifndef QT_NO_DOCKWIDGET
   if (QLayoutItem *ret = dockAreaLayout.takeAt(x, index)) {
      return ret;
   }
#else
   if (centralWidgetItem != nullptr && (*x)++ == index) {
      QLayoutItem *ret  = centralWidgetItem;
      centralWidgetItem = nullptr;
      return ret;
   }
#endif

   return nullptr;
}

QList<int> QMainWindowLayoutState::indexOf(QWidget *widget) const
{
   QList<int> result;

#ifndef QT_NO_TOOLBAR
   // is it a toolbar?
   if (QToolBar *toolBar = qobject_cast<QToolBar *>(widget)) {
      result = toolBarAreaLayout.indexOf(toolBar);
      if (!result.isEmpty()) {
         result.prepend(0);
      }
      return result;
   }
#endif

#ifndef QT_NO_DOCKWIDGET
   // is it a dock widget?

   if (qobject_cast<QDockWidget *>(widget) || qobject_cast<QDockWidgetGroupWindow *>(widget)) {
      result = dockAreaLayout.indexOf(widget);

      if (!result.isEmpty()) {
         result.prepend(1);
      }

      return result;
   }
#endif

   return result;
}

bool QMainWindowLayoutState::contains(QWidget *widget) const
{
#ifndef QT_NO_DOCKWIDGET
   if (dockAreaLayout.centralWidgetItem != nullptr && dockAreaLayout.centralWidgetItem->widget() == widget) {
      return true;
   }

   if (!dockAreaLayout.indexOf(widget).isEmpty()) {
      return true;
   }

#else
   if (centralWidgetItem != nullptr && centralWidgetItem->widget() == widget) {
      return true;
   }
#endif

#ifndef QT_NO_TOOLBAR
   if (! toolBarAreaLayout.indexOf(widget).isEmpty()) {
      return true;
   }
#endif
   return false;
}

void QMainWindowLayoutState::setCentralWidget(QWidget *widget)
{
   QLayoutItem *item = nullptr;

   //make sure we remove the widget
   deleteCentralWidgetItem();

   if (widget != nullptr) {
      item = new QWidgetItemV2(widget);
   }

#ifndef QT_NO_DOCKWIDGET
   dockAreaLayout.centralWidgetItem = item;

#else
   centralWidgetItem = item;

#endif
}

QWidget *QMainWindowLayoutState::centralWidget() const
{
   QLayoutItem *item = nullptr;

#ifndef QT_NO_DOCKWIDGET
   item = dockAreaLayout.centralWidgetItem;
#else
   item = centralWidgetItem;
#endif

   if (item != nullptr) {
      return item->widget();
   }

   return nullptr;
}

QList<int> QMainWindowLayoutState::gapIndex(QWidget *widget, const QPoint &pos) const
{
   QList<int> result;

#ifndef QT_NO_TOOLBAR
   // is it a toolbar?
   if (qobject_cast<QToolBar *>(widget) != nullptr) {
      result = toolBarAreaLayout.gapIndex(pos);

      if (! result.isEmpty()) {
         result.prepend(0);
      }
      return result;
   }
#endif

#ifndef QT_NO_DOCKWIDGET
   // is it a dock widget?
   if (dynamic_cast<QDockWidget *>(widget) != nullptr || qobject_cast<QDockWidgetGroupWindow *>(widget)) {
      result = dockAreaLayout.gapIndex(pos);

      if (!result.isEmpty()) {
         result.prepend(1);
      }
      return result;
   }
#endif

   return result;
}

bool QMainWindowLayoutState::insertGap(const QList<int> &path, QLayoutItem *item)
{
   if (path.isEmpty()) {
      return false;
   }

   int i = path.first();

#ifndef QT_NO_TOOLBAR
   if (i == 0) {
      Q_ASSERT(qobject_cast<QToolBar *>(item->widget()) != nullptr);
      return toolBarAreaLayout.insertGap(path.mid(1), item);
   }
#endif

#ifndef QT_NO_DOCKWIDGET
   if (i == 1) {
      Q_ASSERT(qobject_cast<QDockWidget *>(item->widget()) || qobject_cast<QDockWidgetGroupWindow *>(item->widget()));
      return dockAreaLayout.insertGap(path.mid(1), item);
   }
#endif

   return false;
}

void QMainWindowLayoutState::remove(const QList<int> &path)
{
   int i = path.first();

#ifndef QT_NO_TOOLBAR
   if (i == 0) {
      toolBarAreaLayout.remove(path.mid(1));
   }
#endif

#ifndef QT_NO_DOCKWIDGET
   if (i == 1) {
      dockAreaLayout.remove(path.mid(1));
   }
#endif
}

void QMainWindowLayoutState::remove(QLayoutItem *item)
{
#ifndef QT_NO_TOOLBAR
   toolBarAreaLayout.remove(item);
#endif

#ifndef QT_NO_DOCKWIDGET
   // is it a dock widget?
   if (QDockWidget *dockWidget = qobject_cast<QDockWidget *>(item->widget())) {
      QList<int> path = dockAreaLayout.indexOf(dockWidget);
      if (!path.isEmpty()) {
         dockAreaLayout.remove(path);
      }
   }
#endif
}

void QMainWindowLayoutState::clear()
{
#ifndef QT_NO_TOOLBAR
   toolBarAreaLayout.clear();
#endif

#ifndef QT_NO_DOCKWIDGET
   dockAreaLayout.clear();
#else
   centralWidgetRect = QRect();
#endif

   rect = QRect();
}

bool QMainWindowLayoutState::isValid() const
{
   return rect.isValid();
}

QLayoutItem *QMainWindowLayoutState::item(const QList<int> &path)
{
   int i = path.first();

#ifndef QT_NO_TOOLBAR
   if (i == 0) {
      const QToolBarAreaLayoutItem *tbItem = toolBarAreaLayout.item(path.mid(1));
      Q_ASSERT(tbItem);
      return tbItem->widgetItem;
   }
#endif

#ifndef QT_NO_DOCKWIDGET
   if (i == 1) {
      return dockAreaLayout.item(path.mid(1)).widgetItem;
   }
#endif

   return nullptr;
}

QRect QMainWindowLayoutState::itemRect(const QList<int> &path) const
{
   int i = path.first();

#ifndef QT_NO_TOOLBAR
   if (i == 0) {
      return toolBarAreaLayout.itemRect(path.mid(1));
   }
#endif

#ifndef QT_NO_DOCKWIDGET
   if (i == 1) {
      return dockAreaLayout.itemRect(path.mid(1));
   }
#endif

   return QRect();
}

QRect QMainWindowLayoutState::gapRect(const QList<int> &path) const
{
   int i = path.first();

#ifndef QT_NO_TOOLBAR
   if (i == 0) {
      return toolBarAreaLayout.itemRect(path.mid(1));
   }
#endif

#ifndef QT_NO_DOCKWIDGET
   if (i == 1) {
      return dockAreaLayout.gapRect(path.mid(1));
   }
#endif

   return QRect();
}

QLayoutItem *QMainWindowLayoutState::plug(const QList<int> &path)
{
   int i = path.first();

#ifndef QT_NO_TOOLBAR
   if (i == 0) {
      return toolBarAreaLayout.plug(path.mid(1));
   }
#endif

#ifndef QT_NO_DOCKWIDGET
   if (i == 1) {
      return dockAreaLayout.plug(path.mid(1));
   }
#endif

   return nullptr;
}

QLayoutItem *QMainWindowLayoutState::unplug(const QList<int> &path, QMainWindowLayoutState *other)
{
   int i = path.first();

#ifdef QT_NO_TOOLBAR
   (void) other;
#else
   if (i == 0) {
      return toolBarAreaLayout.unplug(path.mid(1), other ? &other->toolBarAreaLayout : nullptr);
   }
#endif

#ifndef QT_NO_DOCKWIDGET
   if (i == 1) {
      return dockAreaLayout.unplug(path.mid(1));
   }
#endif

   return nullptr;
}

void QMainWindowLayoutState::saveState(QDataStream &stream) const
{
#ifndef QT_NO_DOCKWIDGET
   dockAreaLayout.saveState(stream);

#ifndef QT_NO_TABBAR
   QList<QDockWidgetGroupWindow *> floatingTabs =
            mainWindow->findChildren<QDockWidgetGroupWindow *>(QString(), Qt::FindDirectChildrenOnly);

   for (QDockWidgetGroupWindow *floating : floatingTabs) {
      if (floating->layoutInfo()->isEmpty()) {
         continue;
      }

      stream << uchar(QDockAreaLayout::FloatingDockWidgetTabMarker) << floating->geometry();
      floating->layoutInfo()->saveState(stream);
   }
#endif

#endif

#ifndef QT_NO_TOOLBAR
   toolBarAreaLayout.saveState(stream);
#endif
}

template <typename T>
static QList<T> findChildrenHelper(const QObject *o)
{
   const QObjectList &list = o->children();
   QList<T> result;

   for (int i = 0; i < list.size(); ++i) {
      if (T t = dynamic_cast<T>(list[i])) {
         result.append(t);
      }
   }

   return result;
}

bool QMainWindowLayoutState::checkFormat(QDataStream &stream)
{
   while (! stream.atEnd()) {
      uchar marker;
      stream >> marker;

      switch (marker) {

#ifndef QT_NO_DOCKWIDGET
         case QDockAreaLayout::DockWidgetStateMarker: {
            QList<QDockWidget *> dockWidgets = findChildrenHelper<QDockWidget *>(mainWindow);

            for (QDockWidgetGroupWindow *floating : findChildrenHelper<QDockWidgetGroupWindow *>(mainWindow)) {
               dockWidgets += findChildrenHelper<QDockWidget *>(floating);
            }

            // true indicates test mode
            if (! dockAreaLayout.restoreState(stream, dockWidgets, true)) {
               return false;
            }
         }
         break;

#ifndef QT_NO_TABBAR
         case QDockAreaLayout::FloatingDockWidgetTabMarker: {
            QRect geom;
            stream >> geom;

            QDockAreaLayoutInfo info;
            QList<QDockWidget *> dockWidgets = findChildrenHelper<QDockWidget *>(mainWindow);

            for (QDockWidgetGroupWindow *floating : findChildrenHelper<QDockWidgetGroupWindow *>(mainWindow)) {
               dockWidgets += findChildrenHelper<QDockWidget *>(floating);
            }

            // true indicates test mode
            if (! info.restoreState(stream, dockWidgets, true)) {
               return false;
            }
         }
         break;
#endif

#endif // QT_NO_DOCKWIDGET


#ifndef QT_NO_TOOLBAR
         case QToolBarAreaLayout::ToolBarStateMarker:
         case QToolBarAreaLayout::ToolBarStateMarkerEx: {
            QList<QToolBar *> toolBars = findChildrenHelper<QToolBar *>(mainWindow);

            // true indicates test mode
            if (! toolBarAreaLayout.restoreState(stream, toolBars, marker, true)) {
               return false;
            }
         }
         break;
#endif

         default:
            // there was an error during the parsing
            return false;
      }
   }

   return true;
}

bool QMainWindowLayoutState::restoreState(QDataStream &dataStream, const QMainWindowLayoutState &oldState)
{
   // make a copy of the data so we can read it more than once
   QByteArray copy;

   while (! dataStream.atEnd()) {
      int length = 1024;

      QByteArray ba(length, '\0');
      length = dataStream.readRawData(ba.data(), ba.size());
      ba.resize(length);

      copy += ba;
   }

   QDataStream tmp(copy);

   if (! checkFormat(tmp)) {
      // format unknown
      return false;
   }

   QDataStream stream(copy);

   while (! stream.atEnd()) {
      uchar marker;
      stream >> marker;

      switch (marker) {

#ifndef QT_NO_DOCKWIDGET
         case QDockAreaLayout::DockWidgetStateMarker: {
            QList<QDockWidget *> dockWidgets = findChildrenHelper<QDockWidget *>(mainWindow);

            for (QDockWidgetGroupWindow *floating : findChildrenHelper<QDockWidgetGroupWindow *>(mainWindow)) {
               dockWidgets += findChildrenHelper<QDockWidget *>(floating);
            }

            if (! dockAreaLayout.restoreState(stream, dockWidgets)) {
               return false;
            }

            for (auto item : dockWidgets) {
               QList<int> path = dockAreaLayout.indexOf(item);

               if (path.isEmpty()) {
                  QList<int> oldPath = oldState.dockAreaLayout.indexOf(item);

                  if (oldPath.isEmpty()) {
                     continue;
                  }

                  QDockAreaLayoutInfo *info = dockAreaLayout.info(oldPath);
                  if (info == nullptr) {
                     continue;
                  }

                  info->item_list.append(QDockAreaLayoutItem(new QDockWidgetItem(item)));
               }
            }
         }
         break;

#ifndef QT_NO_TABBAR
         case QDockAreaLayout::FloatingDockWidgetTabMarker: {
            QList<QDockWidget *> dockWidgets = findChildrenHelper<QDockWidget *>(mainWindow);

            for (QDockWidgetGroupWindow *floating : findChildrenHelper<QDockWidgetGroupWindow *>(mainWindow)) {
               dockWidgets += findChildrenHelper<QDockWidget *>(floating);
            }

            QDockWidgetGroupWindow *floatingTab = qt_mainwindow_layout(mainWindow)->createTabbedDockWindow();
            *floatingTab->layoutInfo() = QDockAreaLayoutInfo(&dockAreaLayout.sep, QInternal::LeftDock,
                  Qt::Horizontal, QTabBar::RoundedSouth, mainWindow);

            QRect geometry;
            stream >> geometry;

            if (! floatingTab->layoutInfo()->restoreState(stream, dockWidgets, false)) {
               return false;
            }

            geometry = QDockAreaLayout::constrainedRect(geometry, floatingTab);
            floatingTab->move(geometry.topLeft());
            floatingTab->resize(geometry.size());
            floatingTab->show();
         }
         break;

#endif
#endif // QT_NO_DOCKWIDGET

#ifndef QT_NO_TOOLBAR
         case QToolBarAreaLayout::ToolBarStateMarker:
         case QToolBarAreaLayout::ToolBarStateMarkerEx: {
            QList<QToolBar *> toolBars = findChildrenHelper<QToolBar *>(mainWindow);

            if (! toolBarAreaLayout.restoreState(stream, toolBars, marker)) {
               return false;
            }

            for (int i = 0; i < toolBars.size(); ++i) {
               QToolBar *w = toolBars.at(i);
               QList<int> path = toolBarAreaLayout.indexOf(w);

               if (path.isEmpty()) {
                  QList<int> oldPath = oldState.toolBarAreaLayout.indexOf(w);
                  if (oldPath.isEmpty()) {
                     continue;
                  }

                  toolBarAreaLayout.docks[oldPath.at(0)].insertToolBar(nullptr, w);
               }
            }
         }
         break;
#endif

         default:
            return false;
      }
   }

   return true;
}

#ifndef QT_NO_TOOLBAR

static inline void validateToolBarArea(Qt::ToolBarArea &area)
{
   switch (area) {
      case Qt::LeftToolBarArea:
      case Qt::RightToolBarArea:
      case Qt::TopToolBarArea:
      case Qt::BottomToolBarArea:
         break;

      default:
         area = Qt::TopToolBarArea;
   }
}

static QInternal::DockPosition toDockPos(Qt::ToolBarArea area)
{
   switch (area) {
      case Qt::LeftToolBarArea:
         return QInternal::LeftDock;

      case Qt::RightToolBarArea:
         return QInternal::RightDock;

      case Qt::TopToolBarArea:
         return QInternal::TopDock;

      case Qt::BottomToolBarArea:
         return QInternal::BottomDock;

      default:
         break;
   }

   return QInternal::DockCount;
}

static Qt::ToolBarArea toToolBarArea(QInternal::DockPosition pos)
{
   switch (pos) {
      case QInternal::LeftDock:
         return Qt::LeftToolBarArea;

      case QInternal::RightDock:
         return Qt::RightToolBarArea;

      case QInternal::TopDock:
         return Qt::TopToolBarArea;

      case QInternal::BottomDock:
         return Qt::BottomToolBarArea;

      default:
         break;
   }
   return Qt::NoToolBarArea;
}

static inline Qt::ToolBarArea toToolBarArea(int pos)
{
   return toToolBarArea(static_cast<QInternal::DockPosition>(pos));
}

void QMainWindowLayout::addToolBarBreak(Qt::ToolBarArea area)
{
   validateToolBarArea(area);

   layoutState.toolBarAreaLayout.addToolBarBreak(toDockPos(area));

   if (savedState.isValid()) {
      savedState.toolBarAreaLayout.addToolBarBreak(toDockPos(area));
   }

   invalidate();
}

void QMainWindowLayout::insertToolBarBreak(QToolBar *before)
{
   layoutState.toolBarAreaLayout.insertToolBarBreak(before);

   if (savedState.isValid()) {
      savedState.toolBarAreaLayout.insertToolBarBreak(before);
   }

   invalidate();
}

void QMainWindowLayout::removeToolBarBreak(QToolBar *before)
{
   layoutState.toolBarAreaLayout.removeToolBarBreak(before);

   if (savedState.isValid()) {
      savedState.toolBarAreaLayout.removeToolBarBreak(before);
   }

   invalidate();
}

void QMainWindowLayout::moveToolBar(QToolBar *toolbar, int pos)
{
   layoutState.toolBarAreaLayout.moveToolBar(toolbar, pos);

   if (savedState.isValid()) {
      savedState.toolBarAreaLayout.moveToolBar(toolbar, pos);
   }

   invalidate();
}

void QMainWindowLayout::removeToolBar(QToolBar *toolbar)
{
   if (toolbar) {
      QMainWindow *obj = dynamic_cast<QMainWindow *>(parentWidget());

      if (obj != nullptr) {
         QObject::disconnect(obj, &QMainWindow::iconSizeChanged,        toolbar, &QToolBar::_q_updateIconSize);
         QObject::disconnect(obj, &QMainWindow::toolButtonStyleChanged, toolbar, &QToolBar::_q_updateToolButtonStyle);
      }

      removeWidget(toolbar);
   }
}

void QMainWindowLayout::addToolBar(Qt::ToolBarArea area, QToolBar *toolbar, bool)
{
   validateToolBarArea(area);

   {
      // add the toolbar to the layout
      addChildWidget(toolbar);
      QLayoutItem *item = layoutState.toolBarAreaLayout.addToolBar(toDockPos(area), toolbar);

      if (savedState.isValid() && item) {
         // copy the toolbar also in the saved state
         savedState.toolBarAreaLayout.insertItem(toDockPos(area), item);
      }
      invalidate();

      // this ensures that the toolbar has the right window flags (not floating any more)
      toolbar->d_func()->updateWindowFlags(false /*floating*/);
   }
}

void QMainWindowLayout::insertToolBar(QToolBar *before, QToolBar *toolbar)
{
   addChildWidget(toolbar);
   QLayoutItem *item = layoutState.toolBarAreaLayout.insertToolBar(before, toolbar);

   if (savedState.isValid() && item) {
      // copy the toolbar also in the saved state
      savedState.toolBarAreaLayout.insertItem(before, item);
   }

   if (!currentGapPos.isEmpty() && currentGapPos.first() == 0) {
      currentGapPos = layoutState.toolBarAreaLayout.currentGapIndex();
      if (!currentGapPos.isEmpty()) {
         currentGapPos.prepend(0);
         currentGapRect = layoutState.itemRect(currentGapPos);
      }
   }

   invalidate();
}

Qt::ToolBarArea QMainWindowLayout::toolBarArea(QToolBar *toolbar) const
{
   QInternal::DockPosition pos = layoutState.toolBarAreaLayout.findToolBar(toolbar);

   switch (pos) {
      case QInternal::LeftDock:
         return Qt::LeftToolBarArea;

      case QInternal::RightDock:
         return Qt::RightToolBarArea;

      case QInternal::TopDock:
         return Qt::TopToolBarArea;

      case QInternal::BottomDock:
         return Qt::BottomToolBarArea;

      default:
         break;
   }

   return Qt::NoToolBarArea;
}

bool QMainWindowLayout::toolBarBreak(QToolBar *toolBar) const
{
   return layoutState.toolBarAreaLayout.toolBarBreak(toolBar);
}

void QMainWindowLayout::getStyleOptionInfo(QStyleOptionToolBar *option, QToolBar *toolBar) const
{
   option->toolBarArea = toolBarArea(toolBar);
   layoutState.toolBarAreaLayout.getStyleOptionInfo(option, toolBar);
}

void QMainWindowLayout::toggleToolBarsVisible()
{
   bool updateNonUnifiedParts = true;
   if (updateNonUnifiedParts) {
      layoutState.toolBarAreaLayout.visible = !layoutState.toolBarAreaLayout.visible;
      if (!layoutState.mainWindow->isMaximized()) {
         QPoint topLeft = parentWidget()->geometry().topLeft();
         QRect r = parentWidget()->geometry();
         r = layoutState.toolBarAreaLayout.rectHint(r);
         r.moveTo(topLeft);
         parentWidget()->setGeometry(r);
      } else {
         update();
      }
   }
}

#endif // QT_NO_TOOLBAR


#ifndef QT_NO_DOCKWIDGET

static QInternal::DockPosition toDockPos(Qt::DockWidgetArea area)
{
   switch (area) {
      case Qt::LeftDockWidgetArea:
         return QInternal::LeftDock;

      case Qt::RightDockWidgetArea:
         return QInternal::RightDock;

      case Qt::TopDockWidgetArea:
         return QInternal::TopDock;

      case Qt::BottomDockWidgetArea:
         return QInternal::BottomDock;

      default:
         break;
   }

   return QInternal::DockCount;
}

static Qt::DockWidgetArea toDockWidgetArea(QInternal::DockPosition pos)
{
   switch (pos) {
      case QInternal::LeftDock :
         return Qt::LeftDockWidgetArea;

      case QInternal::RightDock :
         return Qt::RightDockWidgetArea;

      case QInternal::TopDock :
         return Qt::TopDockWidgetArea;

      case QInternal::BottomDock :
         return Qt::BottomDockWidgetArea;

      default:
         break;
   }

   return Qt::NoDockWidgetArea;
}

static inline Qt::DockWidgetArea toDockWidgetArea(int pos)
{
   return toDockWidgetArea(static_cast<QInternal::DockPosition>(pos));
}

void QMainWindowLayout::setCorner(Qt::Corner corner, Qt::DockWidgetArea area)
{
   if (layoutState.dockAreaLayout.corners[corner] == area) {
      return;
   }

   layoutState.dockAreaLayout.corners[corner] = area;

   if (savedState.isValid()) {
      savedState.dockAreaLayout.corners[corner] = area;
   }
   invalidate();
}

Qt::DockWidgetArea QMainWindowLayout::corner(Qt::Corner corner) const
{
   return layoutState.dockAreaLayout.corners[corner];
}

void QMainWindowLayout::addDockWidget(Qt::DockWidgetArea area,
   QDockWidget *dockwidget,
   Qt::Orientation orientation)
{
   addChildWidget(dockwidget);

   // If we are currently moving a separator, then we need to abort the move, since each
   // time we move the mouse layoutState is replaced by savedState modified by the move.
   if (! movingSeparator.isEmpty()) {
      endSeparatorMove(movingSeparatorPos);
   }

   layoutState.dockAreaLayout.addDockWidget(toDockPos(area), dockwidget, orientation);
   emit dockwidget->dockLocationChanged(area);
   invalidate();
}

void QMainWindowLayout::tabifyDockWidget(QDockWidget *first, QDockWidget *second)
{
   addChildWidget(second);
   layoutState.dockAreaLayout.tabifyDockWidget(first, second);
   emit second->dockLocationChanged(dockWidgetArea(first));
   invalidate();
}

bool QMainWindowLayout::restoreDockWidget(QDockWidget *dockwidget)
{
   addChildWidget(dockwidget);
   if (! layoutState.dockAreaLayout.restoreDockWidget(dockwidget)) {
      return false;
   }

   emit dockwidget->dockLocationChanged(dockWidgetArea(dockwidget));
   invalidate();
   return true;
}

#ifndef QT_NO_TABBAR
bool QMainWindowLayout::documentMode() const
{
   return _documentMode;
}

void QMainWindowLayout::setDocumentMode(bool enabled)
{
   if (_documentMode == enabled) {
      return;
   }

   _documentMode = enabled;

   // Update the document mode for all tab bars
   for (QTabBar *bar : usedTabBars) {
      bar->setDocumentMode(_documentMode);
   }

   for (QTabBar *bar : unusedTabBars) {
      bar->setDocumentMode(_documentMode);
   }
}
#endif // QT_NO_TABBAR

void QMainWindowLayout::setVerticalTabsEnabled(bool enabled)
{
#ifdef QT_NO_TABBAR
   (void) enabled;
#else
   if (verticalTabsEnabled == enabled) {
      return;
   }

   verticalTabsEnabled = enabled;

   updateTabBarShapes();
#endif // QT_NO_TABBAR
}

#ifndef QT_NO_TABWIDGET
QTabWidget::TabShape QMainWindowLayout::tabShape() const
{
   return _tabShape;
}

void QMainWindowLayout::setTabShape(QTabWidget::TabShape tabShape)
{
   if (_tabShape == tabShape) {
      return;
   }

   _tabShape = tabShape;

   updateTabBarShapes();
}

QTabWidget::TabPosition QMainWindowLayout::tabPosition(Qt::DockWidgetArea area) const
{
   return tabPositions[toDockPos(area)];
}

void QMainWindowLayout::setTabPosition(Qt::DockWidgetAreas areas, QTabWidget::TabPosition tabPosition)
{
   const Qt::DockWidgetArea dockWidgetAreas[] = {
      Qt::TopDockWidgetArea,
      Qt::LeftDockWidgetArea,
      Qt::BottomDockWidgetArea,
      Qt::RightDockWidgetArea
   };

   const QInternal::DockPosition dockPositions[] = {
      QInternal::TopDock,
      QInternal::LeftDock,
      QInternal::BottomDock,
      QInternal::RightDock
   };

   for (int i = 0; i < QInternal::DockCount; ++i)
      if (areas & dockWidgetAreas[i]) {
         tabPositions[dockPositions[i]] = tabPosition;
      }

   updateTabBarShapes();
}

static inline QTabBar::Shape tabBarShapeFrom(QTabWidget::TabShape shape, QTabWidget::TabPosition position)
{
   const bool rounded = (shape == QTabWidget::Rounded);

   if (position == QTabWidget::North) {
      return rounded ? QTabBar::RoundedNorth : QTabBar::TriangularNorth;
   }

   if (position == QTabWidget::South) {
      return rounded ? QTabBar::RoundedSouth : QTabBar::TriangularSouth;
   }

   if (position == QTabWidget::East) {
      return rounded ? QTabBar::RoundedEast : QTabBar::TriangularEast;
   }

   if (position == QTabWidget::West) {
      return rounded ? QTabBar::RoundedWest : QTabBar::TriangularWest;
   }

   return QTabBar::RoundedNorth;
}
#endif // QT_NO_TABWIDGET

#ifndef QT_NO_TABBAR
void QMainWindowLayout::updateTabBarShapes()
{
#ifndef QT_NO_TABWIDGET
   const QTabWidget::TabPosition vertical[] = {
      QTabWidget::West,
      QTabWidget::East,
      QTabWidget::North,
      QTabWidget::South
   };

#else
   const QTabBar::Shape vertical[] = {
      QTabBar::RoundedWest,
      QTabBar::RoundedEast,
      QTabBar::RoundedNorth,
      QTabBar::RoundedSouth
   };
#endif

   QDockAreaLayout &layout = layoutState.dockAreaLayout;

   for (int i = 0; i < QInternal::DockCount; ++i) {

#ifndef QT_NO_TABWIDGET
      QTabWidget::TabPosition pos = verticalTabsEnabled ? vertical[i] : tabPositions[i];
      QTabBar::Shape shape = tabBarShapeFrom(_tabShape, pos);
#else
      QTabBar::Shape shape = verticalTabsEnabled ? vertical[i] : QTabBar::RoundedSouth;
#endif

      layout.m_docks[i].setTabBarShape(shape);
   }
}
#endif // QT_NO_TABBAR

void QMainWindowLayout::splitDockWidget(QDockWidget *after,
      QDockWidget *dockwidget, Qt::Orientation orientation)
{
   addChildWidget(dockwidget);
   layoutState.dockAreaLayout.splitDockWidget(after, dockwidget, orientation);
   emit dockwidget->dockLocationChanged(dockWidgetArea(after));
   invalidate();
}

Qt::DockWidgetArea QMainWindowLayout::dockWidgetArea(QWidget *widget) const
{
   QList<int> pathToWidget = layoutState.dockAreaLayout.indexOf(widget);

   if (pathToWidget.isEmpty()) {
      return Qt::NoDockWidgetArea;
   }

   return toDockWidgetArea(pathToWidget.first());
}

void QMainWindowLayout::keepSize(QDockWidget *w)
{
   layoutState.dockAreaLayout.keepSize(w);
}

#ifndef QT_NO_TABBAR

class QMainWindowTabBar : public QTabBar
{
   QMainWindow *mainWindow;
   QDockWidget *draggingDock;   // Currently dragging (detached) dock widget

 public:
   QMainWindowTabBar(QMainWindow *parent);

 protected:
   bool event(QEvent *e) override;
   void mouseReleaseEvent(QMouseEvent *) override;
   void mouseMoveEvent(QMouseEvent *) override;

};

QMainWindowTabBar::QMainWindowTabBar(QMainWindow *parent)
   : QTabBar(parent), mainWindow(parent), draggingDock(nullptr)
{
   setExpanding(false);
}

void QMainWindowTabBar::mouseMoveEvent(QMouseEvent *e)
{
   // The QTabBar handles the moving (reordering) of tabs.
   // When QTabBarPrivate::dragInProgress is true, and that the mouse is outside of a region
   // around the QTabBar, we will consider the user wants to drag that QDockWidget away from this
   // tab area.

   QTabBarPrivate *d = static_cast<QTabBarPrivate *>(d_ptr.data());
   if (!draggingDock && (mainWindow->dockOptions() & QMainWindow::GroupedDragging)) {
      int offset = QApplication::startDragDistance() + 1;
      offset *= 3;

      QRect r = rect().adjusted(-offset, -offset, offset, offset);

      if (d->dragInProgress && !r.contains(e->pos()) && d->validIndex(d->pressedIndex)) {
         QMainWindowLayout *mlayout = qt_mainwindow_layout(mainWindow);
         QDockAreaLayoutInfo *info = mlayout->dockInfo(this);
         Q_ASSERT(info);

         int idx = info->tabIndexToListIndex(d->pressedIndex);
         const QDockAreaLayoutItem &item = info->item_list.at(idx);

         if (item.widgetItem
            && (draggingDock = qobject_cast<QDockWidget *>(item.widgetItem->widget()))) {
            // We should drag this QDockWidget away by unpluging it.
            // First cancel the QTabBar move

            d->moveTabFinished(d->pressedIndex);
            d->pressedIndex = -1;

            if (d->movingTab) {
               d->movingTab->setVisible(false);
            }

            d->dragStartPosition = QPoint();

            // Then starts the drag using QDockWidgetPrivate
            QDockWidgetPrivate *dockPriv = static_cast<QDockWidgetPrivate *>(QWidgetPrivate::get(draggingDock));
            QDockWidgetLayout *dwlayout  = static_cast<QDockWidgetLayout *>(draggingDock->layout());

            dockPriv->initDrag(dwlayout->titleArea().center(), true);
            dockPriv->startDrag(false);

            if (dockPriv->state) {
               dockPriv->state->ctrlDrag = e->modifiers() & Qt::ControlModifier;
            }
         }
      }
   }

   if (draggingDock) {
      QDockWidgetPrivate *dockPriv = static_cast<QDockWidgetPrivate *>(QWidgetPrivate::get(draggingDock));
      if (dockPriv->state && dockPriv->state->dragging) {
         QPoint pos = e->globalPos() - dockPriv->state->pressPos;
         draggingDock->move(pos);
         // move will call QMainWindowLayout::hover
      }
   }
   QTabBar::mouseMoveEvent(e);
}

void QMainWindowTabBar::mouseReleaseEvent(QMouseEvent *e)
{
   if (draggingDock && e->button() == Qt::LeftButton) {
      QDockWidgetPrivate *dockPriv = static_cast<QDockWidgetPrivate *>(QWidgetPrivate::get(draggingDock));
      if (dockPriv->state && dockPriv->state->dragging) {
         dockPriv->endDrag();
      }
      draggingDock = nullptr;
   }
   QTabBar::mouseReleaseEvent(e);
}

bool QMainWindowTabBar::event(QEvent *e)
{
   // show the tooltip if tab is too small to fit label

   if (e->type() != QEvent::ToolTip) {
      return QTabBar::event(e);
   }

   QSize size = this->size();
   QSize hint = sizeHint();
   if (shape() == QTabBar::RoundedWest || shape() == QTabBar::RoundedEast) {
      size = size.transposed();
      hint = hint.transposed();
   }

   if (size.width() < hint.width()) {
      return QTabBar::event(e);
   }

   e->accept();

   return true;
}

QTabBar *QMainWindowLayout::getTabBar()
{
   QTabBar *result = nullptr;

   if (! unusedTabBars.isEmpty()) {
      result = unusedTabBars.takeLast();

   } else {
      result = new QMainWindowTabBar(static_cast<QMainWindow *>(parentWidget()));
      result->setDrawBase(true);
      result->setElideMode(Qt::ElideRight);
      result->setDocumentMode(_documentMode);
      result->setMovable(true);

      connect(result, &QTabBar::currentChanged, this, &QMainWindowLayout::tabChanged);
      connect(result, &QTabBar::tabMoved,       this, &QMainWindowLayout::tabMoved);
   }

   usedTabBars.insert(result);
   return result;
}

// Allocates a new separator widget if needed
QWidget *QMainWindowLayout::getSeparatorWidget()
{
   QWidget *result = nullptr;

   if (! unusedSeparatorWidgets.isEmpty()) {
      result = unusedSeparatorWidgets.takeLast();

   } else {
      result = new QWidget(parentWidget());
      result->setAttribute(Qt::WA_MouseNoMask, true);
      result->setAutoFillBackground(false);
      result->setObjectName("qt_qmainwindow_extended_splitter");
   }

   usedSeparatorWidgets.insert(result);
   return result;
}

QDockAreaLayoutInfo *QMainWindowLayout::dockInfo(QWidget *widget)
{
   QDockAreaLayoutInfo *info = layoutState.dockAreaLayout.info(widget);
   if (info) {
      return info;
   }

   for (QDockWidgetGroupWindow *dwgw : parent()->findChildren<QDockWidgetGroupWindow *>(QString(),
         Qt::FindDirectChildrenOnly)) {
      info = dwgw->layoutInfo()->info(widget);

      if (info) {
         return info;
      }
   }

   return nullptr;
}
void QMainWindowLayout::tabChanged()
{
   QTabBar *tb = qobject_cast<QTabBar *>(sender());
   if (tb == nullptr) {
      return;
   }

   QDockAreaLayoutInfo *info = dockInfo(tb);

   if (info == nullptr) {
      return;
   }
   info->apply(false);

   if (QWidget *w = centralWidget()) {
      w->raise();
   }
}

void QMainWindowLayout::tabMoved(int from, int to)
{
   QTabBar *tb = qobject_cast<QTabBar *>(sender());
   Q_ASSERT(tb);

   QDockAreaLayoutInfo *info = dockInfo(tb);
   Q_ASSERT(info);

   info->moveTab(from, to);
}
#endif // QT_NO_TABBAR

bool QMainWindowLayout::startSeparatorMove(const QPoint &pos)
{
   movingSeparator = layoutState.dockAreaLayout.findSeparator(pos);

   if (movingSeparator.isEmpty()) {
      return false;
   }

   layoutState.dockAreaLayout.fallbackToSizeHints = false;

   savedState = layoutState;
   movingSeparatorPos = movingSeparatorOrigin = pos;

   return true;
}

bool QMainWindowLayout::separatorMove(const QPoint &pos)
{
   if (movingSeparator.isEmpty()) {
      return false;
   }
   movingSeparatorPos = pos;
   separatorMoveTimer.start(0, this);
   return true;
}

bool QMainWindowLayout::endSeparatorMove(const QPoint &)
{
   if (movingSeparator.isEmpty()) {
      return false;
   }
   movingSeparator.clear();
   savedState.clear();
   return true;
}

void QMainWindowLayout::raise(QDockWidget *widget)
{
   QDockAreaLayoutInfo *info = dockInfo(widget);
   if (info == nullptr) {
      return;
   }


#ifndef QT_NO_TABBAR
   if (!info->tabbed) {
      return;
   }

   info->setCurrentTab(widget);
#endif
}
#endif // QT_NO_DOCKWIDGET


int QMainWindowLayout::count() const
{
   qWarning("QMainWindowLayout::count() Method is not supported");
   return 0;
}

QLayoutItem *QMainWindowLayout::itemAt(int index) const
{
   int x = 0;

   if (QLayoutItem *ret = layoutState.itemAt(index, &x)) {
      return ret;
   }

   if (statusbar && x++ == index) {
      return statusbar;
   }

   return nullptr;
}

QLayoutItem *QMainWindowLayout::takeAt(int index)
{
   int x = 0;

   if (QLayoutItem *ret = layoutState.takeAt(index, &x)) {

      // the widget might in fact have been destroyed by now
      if (QWidget *w = ret->widget()) {
         widgetAnimator.abort(w);

         if (w == pluggingWidget) {
            pluggingWidget = nullptr;
         }
      }

      if (savedState.isValid() ) {
         //we need to remove the item also from the saved state to prevent crash
         savedState.remove(ret);
         //Also, the item may be contained several times as a gap item.
         layoutState.remove(ret);
      }

#ifndef QT_NO_TOOLBAR
      if (!currentGapPos.isEmpty() && currentGapPos.first() == 0) {
         currentGapPos = layoutState.toolBarAreaLayout.currentGapIndex();
         if (!currentGapPos.isEmpty()) {
            currentGapPos.prepend(0);
            currentGapRect = layoutState.itemRect(currentGapPos);
         }
      }
#endif

      return ret;
   }

   if (statusbar && x++ == index) {
      QLayoutItem *ret = statusbar;
      statusbar = nullptr;
      return ret;
   }

   return nullptr;
}

void QMainWindowLayout::setGeometry(const QRect &_r)
{
   if (savedState.isValid()) {
      return;
   }

   QRect r = _r;

   QLayout::setGeometry(r);

   if (statusbar) {
      QRect sbr(QPoint(r.left(), 0),
         QSize(r.width(), statusbar->heightForWidth(r.width()))
         .expandedTo(statusbar->minimumSize()));
      sbr.moveBottom(r.bottom());
      QRect vr = QStyle::visualRect(parentWidget()->layoutDirection(), _r, sbr);
      statusbar->setGeometry(vr);
      r.setBottom(sbr.top() - 1);
   }

   layoutState.rect = r;
   layoutState.fitLayout();
   applyState(layoutState, false);
}

void QMainWindowLayout::addItem(QLayoutItem *)
{
   qWarning("QMainWindowLayout::addItem() Method is not supported");
}

QSize QMainWindowLayout::sizeHint() const
{
   if (!szHint.isValid()) {
      szHint = layoutState.sizeHint();
      const QSize sbHint = statusbar ? statusbar->sizeHint() : QSize(0, 0);
      szHint = QSize(qMax(sbHint.width(), szHint.width()),
            sbHint.height() + szHint.height());
   }
   return szHint;
}

QSize QMainWindowLayout::minimumSize() const
{
   if (!minSize.isValid()) {
      minSize = layoutState.minimumSize();
      const QSize sbMin = statusbar ? statusbar->minimumSize() : QSize(0, 0);
      minSize = QSize(qMax(sbMin.width(), minSize.width()),
            sbMin.height() + minSize.height());
   }
   return minSize;
}

void QMainWindowLayout::invalidate()
{
   QLayout::invalidate();
   minSize = szHint = QSize();
}

#ifndef QT_NO_DOCKWIDGET
void QMainWindowLayout::setCurrentHoveredFloat(QWidget *w)
{
   if (currentHoveredFloat != w) {
      if (currentHoveredFloat) {
         disconnect(currentHoveredFloat.data(), &QObject::destroyed, this, &QMainWindowLayout::updateGapIndicator);
      }

      currentHoveredFloat = w;

      if (w) {
         connect(w, &QObject::destroyed, this, &QMainWindowLayout::updateGapIndicator, Qt::UniqueConnection);
      }

      updateGapIndicator();
   }
}
#endif

static void fixToolBarOrientation(QLayoutItem *item, int dockPos)
{
#ifndef QT_NO_TOOLBAR
   QToolBar *toolBar = qobject_cast<QToolBar *>(item->widget());
   if (toolBar == nullptr) {
      return;
   }

   QRect oldGeo = toolBar->geometry();

   QInternal::DockPosition pos
      = static_cast<QInternal::DockPosition>(dockPos);
   Qt::Orientation o = pos == QInternal::TopDock || pos == QInternal::BottomDock
      ? Qt::Horizontal : Qt::Vertical;
   if (o != toolBar->orientation()) {
      toolBar->setOrientation(o);
   }

   QSize hint = toolBar->sizeHint().boundedTo(toolBar->maximumSize())
      .expandedTo(toolBar->minimumSize());

   if (toolBar->size() != hint) {
      QRect newGeo(oldGeo.topLeft(), hint);
      if (toolBar->layoutDirection() == Qt::RightToLeft) {
         newGeo.moveRight(oldGeo.right());
      }
      toolBar->setGeometry(newGeo);
   }

#else
   (void) item;
   (void) dockPos;
#endif
}

void QMainWindowLayout::revert(QLayoutItem *widgetItem)
{
   if (!savedState.isValid()) {
      return;
   }

   QWidget *widget = widgetItem->widget();
   layoutState = savedState;
   currentGapPos = layoutState.indexOf(widget);
   if (currentGapPos.isEmpty()) {
      return;
   }
   fixToolBarOrientation(widgetItem, currentGapPos.at(1));
   layoutState.unplug(currentGapPos);
   layoutState.fitLayout();
   currentGapRect = layoutState.itemRect(currentGapPos);

   plug(widgetItem);
}

bool QMainWindowLayout::plug(QLayoutItem *widgetItem)
{
#ifndef QT_NO_DOCKWIDGET
   if (currentHoveredFloat) {
      QWidget *widget = widgetItem->widget();
      QList<int> previousPath = layoutState.indexOf(widget);

      if (! previousPath.isEmpty()) {
         layoutState.remove(previousPath);
      }

      // Let's remove the widget from any possible group window
      for (QDockWidgetGroupWindow *dwgw :
               parent()->findChildren<QDockWidgetGroupWindow *>(QString(), Qt::FindDirectChildrenOnly)) {

         QList<int> path = dwgw->layoutInfo()->indexOf(widget);
         if (!path.isEmpty()) {
            dwgw->layoutInfo()->remove(path);
         }
      }
      currentGapRect = QRect();

      if (QDockWidget *dropTo = qobject_cast<QDockWidget *>(currentHoveredFloat)) {
         //dropping to a normal widget, we mutate it in a QDockWidgetGroupWindow with two tabs
         QDockWidgetGroupWindow *floatingTabs = createTabbedDockWindow();
         floatingTabs->setGeometry(dropTo->geometry());
         QDockAreaLayoutInfo *info = floatingTabs->layoutInfo();
         *info = QDockAreaLayoutInfo(&layoutState.dockAreaLayout.sep, QInternal::LeftDock,
               Qt::Horizontal, QTabBar::RoundedSouth,
               static_cast<QMainWindow *>(parentWidget()));
         info->tabbed = true;
         QLayout *parentLayout = currentHoveredFloat->parentWidget()->layout();
         info->item_list.append(parentLayout->takeAt(parentLayout->indexOf(currentHoveredFloat)));

         dropTo->setParent(floatingTabs);
         dropTo->show();
         dropTo->d_func()->plug(QRect());
         setCurrentHoveredFloat(floatingTabs);
      }

      QDockWidgetGroupWindow *dwgw = qobject_cast<QDockWidgetGroupWindow *>(currentHoveredFloat);
      Q_ASSERT(dwgw);
      Q_ASSERT(dwgw->layoutInfo()->tabbed); // because floating group should always be tabbed
      previousPath = dwgw->layoutInfo()->indexOf(widget);
      if (!previousPath.isEmpty()) {
         dwgw->layoutInfo()->remove(previousPath);
      }
      dwgw->layoutInfo()->tab(0, widgetItem);
      QRect globalRect = dwgw->layoutInfo()->tabContentRect();
      globalRect.moveTopLeft(dwgw->mapToGlobal(globalRect.topLeft()));
      pluggingWidget = widget;
      widgetAnimator.animate(widget, globalRect, dockOptions & QMainWindow::AnimatedDocks);
      return true;
   }
#endif

   if (!parentWidget()->isVisible() || parentWidget()->isMinimized() || currentGapPos.isEmpty()) {
      return false;
   }

   fixToolBarOrientation(widgetItem, currentGapPos.at(1));

   QWidget *widget = widgetItem->widget();

#ifndef QT_NO_DOCKWIDGET
   // remove the widget from any possible group window
   for (QDockWidgetGroupWindow *dwgw :
               parent()->findChildren<QDockWidgetGroupWindow *>(QString(), Qt::FindDirectChildrenOnly)) {

      QList<int> path = dwgw->layoutInfo()->indexOf(widget);
      if (!path.isEmpty()) {
         dwgw->layoutInfo()->remove(path);
      }
   }
#endif

   QList<int> previousPath = layoutState.indexOf(widget);

   const QLayoutItem *it = layoutState.plug(currentGapPos);
   if (!it) {
      return false;
   }

   Q_ASSERT(it == widgetItem);

   if (!previousPath.isEmpty()) {
      layoutState.remove(previousPath);
   }

   pluggingWidget = widget;
   QRect globalRect = currentGapRect;
   globalRect.moveTopLeft(parentWidget()->mapToGlobal(globalRect.topLeft()));

#ifndef QT_NO_DOCKWIDGET
   if (qobject_cast<QDockWidget *>(widget) != nullptr) {

      QDockWidgetLayout *layout = qobject_cast<QDockWidgetLayout *>(widget->layout());
      if (layout->nativeWindowDeco()) {
         globalRect.adjust(0, layout->titleHeight(), 0, 0);
      } else {
         int fw = widget->style()->pixelMetric(QStyle::PM_DockWidgetFrameWidth, nullptr, widget);
         globalRect.adjust(-fw, -fw, fw, fw);
      }
   }
#endif
   widgetAnimator.animate(widget, globalRect, dockOptions & QMainWindow::AnimatedDocks);

   return true;
}

void QMainWindowLayout::animationFinished(QWidget *widget)
{
   //this function is called from within the Widget Animator whenever an animation is finished
   //on a certain widget
#ifndef QT_NO_TOOLBAR
   if (QToolBar *tb = qobject_cast<QToolBar *>(widget)) {
      QToolBarLayout *tbl = qobject_cast<QToolBarLayout *>(tb->layout());

      if (tbl->animating) {
         tbl->animating = false;
         if (tbl->expanded) {
            tbl->layoutActions(tb->size());
         }
         tb->update();
      }
   }
#endif

   if (widget == pluggingWidget) {

#ifndef QT_NO_DOCKWIDGET
      if (QDockWidgetGroupWindow *dwgw = qobject_cast<QDockWidgetGroupWindow *>(widget)) {
         // When the animated widget was a QDockWidgetGroupWindow, it means each of the
         // embedded QDockWidget needs to be plugged back into the QMainWindow layout.
         savedState.clear();
         QDockAreaLayoutInfo *info = dwgw->layoutInfo();
         QDockAreaLayoutInfo *parentInfo;
         QList<int> path;

         if (QDockWidgetGroupWindow *dropTo = qobject_cast<QDockWidgetGroupWindow *>(currentHoveredFloat)) {
            parentInfo = dropTo->layoutInfo();
            Q_ASSERT(parentInfo->tabbed);

            path = parentInfo->indexOf(widget);
            Q_ASSERT(path.size() == 1);

         } else {
            path = layoutState.dockAreaLayout.indexOf(widget);
            Q_ASSERT(path.size() >= 2);
            parentInfo = layoutState.dockAreaLayout.info(path);
            Q_ASSERT(parentInfo);
         }

         if (parentInfo->tabbed) {
            // merge the two tab widgets
            int idx = path.last();
            Q_ASSERT(parentInfo->item_list[idx].widgetItem->widget() == dwgw);
            delete parentInfo->item_list[idx].widgetItem;
            parentInfo->item_list.removeAt(idx);
            std::copy(info->item_list.cbegin(), info->item_list.cend(),
               std::inserter(parentInfo->item_list, parentInfo->item_list.begin() + idx));
            quintptr currentId = info->currentTabId();
            *info = QDockAreaLayoutInfo();
            parentInfo->reparentWidgets(currentHoveredFloat ? currentHoveredFloat.data() : parentWidget());
            parentInfo->updateTabBar();
            parentInfo->setCurrentTabId(currentId);

         } else {
            QDockAreaLayoutItem &item = layoutState.dockAreaLayout.item(path);
            Q_ASSERT(item.widgetItem->widget() == dwgw);

            delete item.widgetItem;
            item.widgetItem = nullptr;

            item.subinfo = new QDockAreaLayoutInfo(std::move(*info));
            *info = QDockAreaLayoutInfo();
            item.subinfo->reparentWidgets(parentWidget());
            item.subinfo->setTabBarShape(parentInfo->tabBarShape);
         }
         dwgw->destroyOrHideIfEmpty();
      }
      if (QDockWidget *dw = qobject_cast<QDockWidget *>(widget)) {
         dw->setParent(currentHoveredFloat ? currentHoveredFloat.data() : parentWidget());
         dw->show();
         dw->d_func()->plug(currentGapRect);
      }
#endif

#ifndef QT_NO_TOOLBAR
      if (QToolBar *tb = qobject_cast<QToolBar *>(widget)) {
         tb->d_func()->plug(currentGapRect);
      }
#endif

      savedState.clear();
      currentGapPos.clear();
      pluggingWidget = nullptr;
      setCurrentHoveredFloat(nullptr);

      //applying the state will make sure that the currentGap is updated correctly
      //and all the geometries (especially the one from the central widget) is correct
      layoutState.apply(false);

#ifndef QT_NO_DOCKWIDGET
#ifndef QT_NO_TABBAR
      if (qobject_cast<QDockWidget *>(widget) != nullptr) {
         // info() might return null if the widget is destroyed while
         // animating but before the animationFinished signal is received.

         if (QDockAreaLayoutInfo *info = dockInfo(widget)) {
            info->setCurrentTab(widget);
         }
      }
#endif
#endif
   }

   if (! widgetAnimator.animating()) {
      //all animations are finished

#ifndef QT_NO_DOCKWIDGET
      parentWidget()->update(layoutState.dockAreaLayout.separatorRegion());

#ifndef QT_NO_TABBAR
      for (QTabBar *tab_bar : usedTabBars) {
         tab_bar->show();
      }
#endif

#endif // QT_NO_DOCKWIDGET
   }

   updateGapIndicator();
}

void QMainWindowLayout::restore(bool keepSavedState)
{
   if (!savedState.isValid()) {
      return;
   }

   layoutState = savedState;
   applyState(layoutState);

   if (!keepSavedState) {
      savedState.clear();
   }

   currentGapPos.clear();
   pluggingWidget = nullptr;
   updateGapIndicator();
}

QMainWindowLayout::QMainWindowLayout(QMainWindow *mainwindow, QLayout *parentLayout)
   : QLayout(parentLayout ? static_cast<QWidget *>(nullptr) : mainwindow), layoutState(mainwindow),
     savedState(mainwindow), dockOptions(QMainWindow::AnimatedDocks | QMainWindow::AllowTabbedDocks),
     statusbar(nullptr)

#ifndef QT_NO_DOCKWIDGET
#ifndef QT_NO_TABBAR
   , _documentMode(false)
   , verticalTabsEnabled(false)
#ifndef QT_NO_TABWIDGET
   , _tabShape(QTabWidget::Rounded)
#endif
#endif

#endif // QT_NO_DOCKWIDGET

   , widgetAnimator(this), pluggingWidget(nullptr)
{
   if (parentLayout) {
      setParent(parentLayout);
   }

#ifndef QT_NO_DOCKWIDGET

#ifndef QT_NO_TABBAR
   sep = mainwindow->style()->pixelMetric(QStyle::PM_DockWidgetSeparatorExtent, nullptr, mainwindow);
#endif

#ifndef QT_NO_TABWIDGET
   for (int i = 0; i < QInternal::DockCount; ++i) {
      tabPositions[i] = QTabWidget::South;
   }
#endif

#endif // QT_NO_DOCKWIDGET

   pluggingWidget = nullptr;

   setObjectName(mainwindow->objectName() + "_layout");
}

QMainWindowLayout::~QMainWindowLayout()
{
   layoutState.deleteAllLayoutItems();
   layoutState.deleteCentralWidgetItem();

   delete statusbar;
}

void QMainWindowLayout::setDockOptions(QMainWindow::DockOptions opts)
{
   if (opts == dockOptions) {
      return;
   }

   dockOptions = opts;

#ifndef QT_NO_DOCKWIDGET
   setVerticalTabsEnabled(opts & QMainWindow::VerticalTabs);
#endif

   invalidate();
}

#ifndef QT_NO_STATUSBAR
QStatusBar *QMainWindowLayout::statusBar() const
{
   return statusbar ? qobject_cast<QStatusBar *>(statusbar->widget()) : nullptr;
}

void QMainWindowLayout::setStatusBar(QStatusBar *sb)
{
   if (sb) {
      addChildWidget(sb);
   }

   delete statusbar;
   statusbar = sb ? new QWidgetItemV2(sb) : nullptr;
   invalidate();
}
#endif // QT_NO_STATUSBAR

QWidget *QMainWindowLayout::centralWidget() const
{
   return layoutState.centralWidget();
}

void QMainWindowLayout::setCentralWidget(QWidget *widget)
{
   if (widget != nullptr) {
      addChildWidget(widget);
   }

   layoutState.setCentralWidget(widget);

   if (savedState.isValid()) {

#ifndef QT_NO_DOCKWIDGET
      savedState.dockAreaLayout.centralWidgetItem = layoutState.dockAreaLayout.centralWidgetItem;
      savedState.dockAreaLayout.fallbackToSizeHints = true;
#else
      savedState.centralWidgetItem = layoutState.centralWidgetItem;
#endif

   }

   invalidate();
}

QLayoutItem *QMainWindowLayout::unplug(QWidget *widget, bool group)
{
#if !defined(QT_NO_DOCKWIDGET) && !defined(QT_NO_TABBAR)
   if (!widget->isWindow() && qobject_cast<const QDockWidgetGroupWindow *>(widget->parentWidget())) {
      if (group) {
         // We are just dragging a floating window as it, not need to do anything, we just have to
         // look up the corresponding QWidgetItem* if it exists
         QList<int> tabbedWindowPath = layoutState.indexOf(widget->parentWidget());
         return tabbedWindowPath.isEmpty() ? nullptr : layoutState.item(tabbedWindowPath);

      } else {
         // We are unplugging a dock widget from a floating window.
         if (QDockWidget *dw = qobject_cast<QDockWidget *>(widget)) {
            dw->d_func()->unplug(widget->geometry());
            int index = widget->parentWidget()->layout()->indexOf(widget);
            return widget->parentWidget()->layout()->itemAt(index);
         }
      }
   }
#endif

   QList<int> path = layoutState.indexOf(widget);
   if (path.isEmpty()) {
      return nullptr;
   }

   QLayoutItem *item = layoutState.item(path);
   if (widget->isWindow()) {
      return item;
   }

   QRect r = layoutState.itemRect(path);
   savedState = layoutState;

#ifndef QT_NO_DOCKWIDGET
   if (QDockWidget *dw = qobject_cast<QDockWidget *>(widget)) {
      Q_ASSERT(path.first() == 1);

      bool actualGroup = false;

#ifndef QT_NO_TABBAR
      if (group && (dockOptions & QMainWindow::GroupedDragging) && path.size() > 3) {
         QDockAreaLayoutItem &parentItem = layoutState.dockAreaLayout.item(path.mid(1, path.size() - 2));
         if (parentItem.subinfo && parentItem.subinfo->tabbed) {
            // The QDockWidget is part of a group of tab and we need to unplug them all.
            actualGroup = true;
            path.removeLast();

            QDockWidgetGroupWindow *floatingTabs = createTabbedDockWindow();
            QDockAreaLayoutInfo *info = floatingTabs->layoutInfo();
            *info = std::move(*parentItem.subinfo);

            delete parentItem.subinfo;
            parentItem.subinfo = nullptr;

            floatingTabs->setGeometry(info->m_dockAreaInfoRect.translated(parentWidget()->pos()));
            floatingTabs->show();
            floatingTabs->raise();
            item = new QDockWidgetGroupWindowItem(floatingTabs);
            parentItem.widgetItem = item;
            savedState = layoutState;
         }
      }
#endif // QT_NO_TABBAR

      if (!actualGroup) {
         dw->d_func()->unplug(r);
      }
   }
#endif

#ifndef QT_NO_TOOLBAR
   if (QToolBar *tb = qobject_cast<QToolBar *>(widget)) {
      tb->d_func()->unplug(r);
   }
#endif


   layoutState.unplug(path, &savedState);
   savedState.fitLayout();
   currentGapPos = path;
   currentGapRect = r;
   updateGapIndicator();

   fixToolBarOrientation(item, currentGapPos.at(1));

   return item;
}

void QMainWindowLayout::updateGapIndicator()
{
#ifndef QT_NO_RUBBERBAND
   if ((! widgetAnimator.animating() && ! currentGapPos.isEmpty()) || currentHoveredFloat) {
      QWidget *expectedParent = currentHoveredFloat ? currentHoveredFloat.data() : parentWidget();

      if (! gapIndicator) {
         gapIndicator = new QRubberBand(QRubberBand::Rectangle, expectedParent);
         gapIndicator->setObjectName("qt_rubberband");

      } else if (gapIndicator->parent() != expectedParent) {
         gapIndicator->setParent(expectedParent);
      }

      gapIndicator->setGeometry(currentHoveredFloat ? currentHoveredFloat->rect() : currentGapRect);
      gapIndicator->show();
      gapIndicator->raise();

   } else if (gapIndicator) {
      gapIndicator->hide();
   }
#endif
}

void QMainWindowLayout::hover(QLayoutItem *widgetItem, const QPoint &mousePos)
{
   if (!parentWidget()->isVisible() || parentWidget()->isMinimized()
            || pluggingWidget != nullptr || widgetItem == nullptr) {
      return;
   }

   QWidget *widget = widgetItem->widget();

#ifndef QT_NO_DOCKWIDGET
   if ((dockOptions & QMainWindow::GroupedDragging) && (qobject_cast<QDockWidget *>(widget)
         || qobject_cast<QDockWidgetGroupWindow *>(widget))) {

      // Check if we are over another floating dock widget
      QVarLengthArray<QWidget *, 10> candidates;

      for (QObject *childWidget : parentWidget()->children()) {
         QWidget *w = qobject_cast<QWidget *>(childWidget);

         if (! w) {
            continue;
         }

         if (! qobject_cast<QDockWidget *>(w) && ! qobject_cast<QDockWidgetGroupWindow *>(w)) {
            continue;
         }

         if (w != widget && w->isTopLevel() && w->isVisible() && ! w->isMinimized()) {
            candidates << w;
         }

         if (QDockWidgetGroupWindow *group = qobject_cast<QDockWidgetGroupWindow *>(w)) {
            // Sometimes, there are floating QDockWidget that have a QDockWidgetGroupWindow as a parent.

            for (QObject *childGroup : group->children()) {
               if (QDockWidget *dw = qobject_cast<QDockWidget *>(childGroup)) {
                  if (dw != widget && dw->isFloating() && dw->isVisible() && ! dw->isMinimized()) {
                     candidates << dw;
                  }
               }
            }
         }
      }

      for (QWidget *w : candidates) {
         QWindow *handle1 = widget->windowHandle();
         QWindow *handle2 = w->windowHandle();

         if (handle1 && handle2 && handle1->screen() != handle2->screen()) {
            continue;
         }

         if (!w->geometry().contains(mousePos)) {
            continue;
         }

         setCurrentHoveredFloat(w);
         restore(true);
         return;
      }
   }
   setCurrentHoveredFloat(nullptr);
#endif //QT_NO_DOCKWIDGET

   QPoint pos = parentWidget()->mapFromGlobal(mousePos);

   if (! savedState.isValid()) {
      savedState = layoutState;
   }

   QList<int> path = savedState.gapIndex(widget, pos);

   if (!path.isEmpty()) {
      bool allowed = false;

#ifndef QT_NO_DOCKWIDGET
      if (QDockWidget *dw = qobject_cast<QDockWidget *>(widget)) {
         allowed = dw->isAreaAllowed(toDockWidgetArea(path.at(1)));
      }

      if (qobject_cast<QDockWidgetGroupWindow *>(widget)) {
         allowed = true;
      }
#endif

#ifndef QT_NO_TOOLBAR
      if (QToolBar *tb = qobject_cast<QToolBar *>(widget)) {
         allowed = tb->isAreaAllowed(toToolBarArea(path.at(1)));
      }
#endif

      if (! allowed) {
         path.clear();
      }
   }

   if (path == currentGapPos) {
      return;     // the gap is already there
   }

   currentGapPos = path;
   if (path.isEmpty()) {
      fixToolBarOrientation(widgetItem, 2); // 2 = top dock, ie. horizontal
      restore(true);
      return;
   }

   fixToolBarOrientation(widgetItem, currentGapPos.at(1));

   QMainWindowLayoutState newState = savedState;

   if (!newState.insertGap(path, widgetItem)) {
      restore(true); // not enough space
      return ;
   }

   QSize min = newState.minimumSize();
   QSize size = newState.rect.size();

   if (min.width() > size.width() || min.height() > size.height()) {
      restore(true);
      return;
   }

   newState.fitLayout();

   currentGapRect = newState.gapRect(currentGapPos);

#ifndef QT_NO_DOCKWIDGET
   parentWidget()->update(layoutState.dockAreaLayout.separatorRegion());
#endif
   layoutState = newState;
   applyState(layoutState);

   updateGapIndicator();
}

QDockWidgetGroupWindow *QMainWindowLayout::createTabbedDockWindow()
{
   QDockWidgetGroupWindow *f = new QDockWidgetGroupWindow(parentWidget(), Qt::Tool);
   new QDockWidgetGroupLayout(f);
   return f;
}

void QMainWindowLayout::applyState(QMainWindowLayoutState &newState, bool animate)
{
#ifndef QT_NO_DOCKWIDGET
#ifndef QT_NO_TABBAR
   QSet<QTabBar *> used = newState.dockAreaLayout.usedTabBars();

   for (QDockWidgetGroupWindow *dwgw : parent()->findChildren<QDockWidgetGroupWindow *>(QString(), Qt::FindDirectChildrenOnly)) {
      used += dwgw->layoutInfo()->usedTabBars();
   }
   QSet<QTabBar *> retired = usedTabBars - used;
   usedTabBars = used;

   for (QTabBar *tab_bar : retired) {
      tab_bar->hide();

      while (tab_bar->count() > 0) {
         tab_bar->removeTab(0);
      }
      unusedTabBars.append(tab_bar);
   }

   if (sep == 1) {
      QSet<QWidget *> usedSeps = newState.dockAreaLayout.usedSeparatorWidgets();
      QSet<QWidget *> retiredSeps = usedSeparatorWidgets - usedSeps;
      usedSeparatorWidgets = usedSeps;

      for (QWidget *sepWidget : retiredSeps) {
         unusedSeparatorWidgets.append(sepWidget);
      }
   }

   for (int i = 0; i < QInternal::DockCount; ++i) {
      newState.dockAreaLayout.m_docks[i].reparentWidgets(parentWidget());
   }

#endif // QT_NO_TABBAR
#endif // QT_NO_DOCKWIDGET
   newState.apply(dockOptions & QMainWindow::AnimatedDocks && animate);
}

void QMainWindowLayout::saveState(QDataStream &stream) const
{
   layoutState.saveState(stream);
}

bool QMainWindowLayout::restoreState(QDataStream &stream)
{
   savedState = layoutState;
   layoutState.clear();
   layoutState.rect = savedState.rect;

   if (!layoutState.restoreState(stream, savedState)) {
      layoutState.deleteAllLayoutItems();
      layoutState = savedState;

      if (parentWidget()->isVisible()) {
         applyState(layoutState, false);   // hides tabBars allocated by newState
      }

      return false;
   }

   if (parentWidget()->isVisible()) {
      layoutState.fitLayout();
      applyState(layoutState, false);
   }

   savedState.deleteAllLayoutItems();
   savedState.clear();

#ifndef QT_NO_DOCKWIDGET
   if (parentWidget()->isVisible()) {

#ifndef QT_NO_TABBAR
      for (QTabBar *tab_bar : usedTabBars) {
         tab_bar->show();
      }
#endif

   }

#endif // QT_NO_DOCKWIDGET

   return true;
}

// Returns if this toolbar *should* be using HIToolbar. Won't work for all in between cases
// for example, you have a toolbar in the top area and then you suddenly turn on HIToolbar.

bool QMainWindowLayout::usesHIToolBar(QToolBar *toolbar) const
{
   (void) toolbar;
   return false;
}

void QMainWindowLayout::timerEvent(QTimerEvent *e)
{
#ifndef QT_NO_DOCKWIDGET
   if (e->timerId() == separatorMoveTimer.timerId()) {
      //let's move the separators
      separatorMoveTimer.stop();
      if (movingSeparator.isEmpty()) {
         return;
      }
      if (movingSeparatorOrigin == movingSeparatorPos) {
         return;
      }

      //when moving the separator, we need to update the previous position
      parentWidget()->update(layoutState.dockAreaLayout.separatorRegion());

      layoutState = savedState;
      layoutState.dockAreaLayout.separatorMove(movingSeparator, movingSeparatorOrigin,
         movingSeparatorPos);
      movingSeparatorPos = movingSeparatorOrigin;
   }
#endif

   QLayout::timerEvent(e);
}

#endif // QT_NO_MAINWINDOW
