/***********************************************************************
*
* Copyright (c) 2012-2026 Barbara Geller
* Copyright (c) 2012-2026 Ansel Sermersheim
*
* Copyright (c) 2015 The Qt Company Ltd.
* Copyright (c) 2012-2016 Digia Plc and/or its subsidiary(-ies).
* Copyright (c) 2008-2012 Nokia Corporation and/or its subsidiary(-ies).
*
* This file is part of CopperSpice.
*
* CopperSpice is free software. You can redistribute it and/or
* modify it under the terms of the GNU Lesser General Public License
* version 2.1 as published by the Free Software Foundation.
*
* CopperSpice is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
*
* https://www.gnu.org/licenses/
*
***********************************************************************/

/* This is the default style sheet.

   IMPORTANT: This style sheet is primarily meant for defining feature
   capablities of styles. Do NOT add default styling rules here. When in
   doubt ask the stylesheet maintainer.

   The stylesheet in here used to be in a CSS file, but was moved here to
   avoid parsing overhead.
*/

#include <qcssparser_p.h>
#include <qstylesheetstyle_p.h>

#ifndef QT_NO_STYLE_STYLESHEET

using namespace QCss;

// This is the class name of the selector.
// Use an empty string where you would use '*' in CSS.
// Ex. QHeaderView

#define SET_ELEMENT_NAME(x) \
    bSelector.elementName = (x)

// This acts as both pseudo state and sub control. The first parameter is the
// string name, and the second is the PseudoClass_* constant.
// The sub control specifier is always the first, and has the type
// PseudoClass_Unknown.
// If there is no PseudoClass_Unknown as the first pseudo, it is assumed to be
// a pseudo state.
// Ex. QComboBox::drop-down:enabled
//                   ^         ^

#define ADD_PSEUDO(x, y) \
    pseudo.type = (y); \
    pseudo.name = (x); \
    bSelector.pseudos << pseudo

// This is attributes. The third parameter is AttributeSelector::*
// Ex. QComboBox[style="QWindowsXPStyle"]
//                 ^           ^

#define ADD_ATTRIBUTE_SELECTOR(x, y, z) \
    attr.name = (x); \
    attr.value = (y); \
    attr.valueMatchCriterium = (z); \
    bSelector.attributeSelectors << attr

// Adds the current basic selector to the rule.
// Several basic selectors behave as AND (space in CSS).

#define ADD_BASIC_SELECTOR \
    selector.basicSelectors << bSelector; \
    bSelector.ids.clear(); \
    bSelector.pseudos.clear(); \
    bSelector.attributeSelectors.clear()

// Adds the current selector to the rule.
// Several selectors behave as OR (comma in CSS).

#define ADD_SELECTOR \
    styleRule.selectors << selector; \
    selector.basicSelectors.clear()

// Sets the name of a property.
// Ex. background: red;
//         ^

#define SET_PROPERTY(x, y) \
    decl.d->property = (x); \
    decl.d->propertyId = (y)

// Adds a value to the current property.
// The first parameter should be Value::KnownIdentifier if the value can be
// found among the Value_* constants, in which case the second should be that
// constant. Otherwise the first parameter is Value::Identifier and the second
// is a string.
// Adding more values is the same as seperating by spaces in CSS.
// Ex. border: 2px solid black;
//              ^    ^     ^

#define ADD_VALUE(x, y) \
    value.type = (x); \
    value.variant = (y); \
    decl.d->values << value

// Adds the current declaration to the rule.
// Ex. border: 2px solid black;
//     \----------------------/

#define ADD_DECLARATION \
    styleRule.declarations << decl; \
    decl.d.detach(); \
    decl.d->values.clear()

// Adds the rule to the stylesheet.
// Use at the end of every CSS block.

#define ADD_STYLE_RULE \
    sheet.styleRules << styleRule; \
    styleRule.selectors.clear(); \
    styleRule.declarations.clear()

StyleSheet QStyleSheetStyle::getDefaultStyleSheet() const
{
   StyleSheet sheet;
   StyleRule styleRule;
   BasicSelector bSelector;
   Selector selector;
   Declaration decl;
   QCss::Value value;
   Pseudo pseudo;
   AttributeSelector attr;

   // pixmap based style does not support any features
   bool styleIsPixmapBased = baseStyle()->inherits("QMacStyle")
      || baseStyle()->inherits("QWindowsXPStyle") || baseStyle()->inherits("QGtkStyle");

   /*QLineEdit {
       -qt-background-role: base;
       border: native;
       -qt-style-features: background-color;
   }*/
   {
      SET_ELEMENT_NAME(QLatin1String("QLineEdit"));
      ADD_BASIC_SELECTOR;
      ADD_SELECTOR;

      SET_PROPERTY(QLatin1String("-qt-background-role"), QtBackgroundRole);
      ADD_VALUE(Value::KnownIdentifier, Value_Base);
      ADD_DECLARATION;

      SET_PROPERTY(QLatin1String("border"), Border);
      ADD_VALUE(Value::KnownIdentifier, Value_Native);
      ADD_DECLARATION;

      SET_PROPERTY(QLatin1String("-qt-style-features"), QtStyleFeatures);
      ADD_VALUE(Value::Identifier, QString::fromLatin1("background-color"));
      ADD_DECLARATION;

      ADD_STYLE_RULE;
   }

   /*QLineEdit:no-frame {
       border: none;
   }*/
   {
      SET_ELEMENT_NAME(QLatin1String("QLineEdit"));
      ADD_PSEUDO(QLatin1String("no-frame"), PseudoClass_Frameless);
      ADD_BASIC_SELECTOR;
      ADD_SELECTOR;

      SET_PROPERTY(QLatin1String("border"), Border);
      ADD_VALUE(Value::KnownIdentifier, Value_None);
      ADD_DECLARATION;

      ADD_STYLE_RULE;
   }

   /*QFrame {
       border: native;
   }*/
   {
      SET_ELEMENT_NAME(QLatin1String("QFrame"));
      ADD_BASIC_SELECTOR;
      ADD_SELECTOR;

      SET_PROPERTY(QLatin1String("border"), Border);
      ADD_VALUE(Value::KnownIdentifier, Value_Native);
      ADD_DECLARATION;

      ADD_STYLE_RULE;
   }

   /*QLabel, QToolBox {
       background: none;
       border-image: none;
   }*/
   {
      SET_ELEMENT_NAME(QLatin1String("QLabel"));
      ADD_BASIC_SELECTOR;
      ADD_SELECTOR;

      SET_ELEMENT_NAME(QLatin1String("QToolBox"));
      ADD_BASIC_SELECTOR;
      ADD_SELECTOR;

      SET_PROPERTY(QLatin1String("background"), Background);
      ADD_VALUE(Value::KnownIdentifier, Value_None);
      ADD_DECLARATION;

      SET_PROPERTY(QLatin1String("border-image"), BorderImage);
      ADD_VALUE(Value::KnownIdentifier, Value_None);
      ADD_DECLARATION;

      ADD_STYLE_RULE;
   }

   /*QGroupBox {
       border: native;
   }*/
   {
      SET_ELEMENT_NAME(QLatin1String("QGroupBox"));
      ADD_BASIC_SELECTOR;
      ADD_SELECTOR;

      SET_PROPERTY(QLatin1String("border"), Border);
      ADD_VALUE(Value::KnownIdentifier, Value_Native);
      ADD_DECLARATION;

      ADD_STYLE_RULE;
   }


   /*QToolTip {
       -qt-background-role: window;
       border: native;
   }*/
   {
      SET_ELEMENT_NAME(QLatin1String("QToolTip"));
      ADD_BASIC_SELECTOR;
      ADD_SELECTOR;

      SET_PROPERTY(QLatin1String("-qt-background-role"), QtBackgroundRole);
      ADD_VALUE(Value::KnownIdentifier, Value_Window);
      ADD_DECLARATION;

      SET_PROPERTY(QLatin1String("border"), Border);
      ADD_VALUE(Value::KnownIdentifier, Value_Native);
      ADD_DECLARATION;

      ADD_STYLE_RULE;
   }

   /*QPushButton, QToolButton {
       border-style: native;
       -qt-style-features: background-color;  //only for not pixmap based styles
   }*/
   {
      SET_ELEMENT_NAME(QLatin1String("QPushButton"));
      ADD_BASIC_SELECTOR;
      ADD_SELECTOR;

      SET_ELEMENT_NAME(QLatin1String("QToolButton"));
      ADD_BASIC_SELECTOR;
      ADD_SELECTOR;

      SET_PROPERTY(QLatin1String("border-style"), BorderStyles);
      ADD_VALUE(Value::KnownIdentifier, Value_Native);
      ADD_DECLARATION;

      if (!styleIsPixmapBased) {
         SET_PROPERTY(QLatin1String("-qt-style-features"), QtStyleFeatures);
         ADD_VALUE(Value::Identifier, QString::fromLatin1("background-color"));
         ADD_DECLARATION;
      }


      ADD_STYLE_RULE;
   }


   /*QComboBox {
       border: native;
       -qt-style-features: background-color background-gradient;   //only for not pixmap based styles
       -qt-background-role: base;
   }*/

   {
      SET_ELEMENT_NAME(QLatin1String("QComboBox"));
      ADD_BASIC_SELECTOR;
      ADD_SELECTOR;

      SET_PROPERTY(QLatin1String("border"), Border);
      ADD_VALUE(Value::KnownIdentifier, Value_Native);
      ADD_DECLARATION;

      if (!styleIsPixmapBased) {
         SET_PROPERTY(QLatin1String("-qt-style-features"), QtStyleFeatures);
         ADD_VALUE(Value::Identifier, QString::fromLatin1("background-color"));
         ADD_VALUE(Value::Identifier, QString::fromLatin1("background-gradient"));
         ADD_DECLARATION;
      }

      SET_PROPERTY(QLatin1String("-qt-background-role"), QtBackgroundRole);
      ADD_VALUE(Value::KnownIdentifier, Value_Base);
      ADD_DECLARATION;

      ADD_STYLE_RULE;
   }


   /*
    QComboBox[style="QFusionStyle"][readOnly="true"],
    {
        -qt-background-role: button;
    }*/

   if ( baseStyle()->inherits("QFusionStyle")) {
      SET_ELEMENT_NAME(QLatin1String("QComboBox"));
      ADD_ATTRIBUTE_SELECTOR(QLatin1String("readOnly"), QLatin1String("true"), AttributeSelector::MatchEqual);
      ADD_BASIC_SELECTOR;
      ADD_SELECTOR;

      SET_PROPERTY(QLatin1String("-qt-background-role"), QtBackgroundRole);
      ADD_VALUE(Value::KnownIdentifier, Value_Button);
      ADD_DECLARATION;

      ADD_STYLE_RULE;
   }

   /*QAbstractSpinBox {
       border: native;
       -qt-style-features: background-color;
       -qt-background-role: base;
   }*/
   {
      SET_ELEMENT_NAME(QLatin1String("QAbstractSpinBox"));
      ADD_BASIC_SELECTOR;
      ADD_SELECTOR;

      SET_PROPERTY(QLatin1String("border"), Border);
      ADD_VALUE(Value::KnownIdentifier, Value_Native);
      ADD_DECLARATION;

      SET_PROPERTY(QLatin1String("-qt-style-features"), QtStyleFeatures);
      ADD_VALUE(Value::Identifier, QString::fromLatin1("background-color"));
      ADD_DECLARATION;

      SET_PROPERTY(QLatin1String("-qt-background-role"), QtBackgroundRole);
      ADD_VALUE(Value::KnownIdentifier, Value_Base);
      ADD_DECLARATION;

      ADD_STYLE_RULE;
   }

   /*QMenu {
       -qt-background-role: window;
   }*/
   {
      SET_ELEMENT_NAME(QLatin1String("QMenu"));
      ADD_BASIC_SELECTOR;
      ADD_SELECTOR;

      SET_PROPERTY(QLatin1String("-qt-background-role"), QtBackgroundRole);
      ADD_VALUE(Value::KnownIdentifier, Value_Window);
      ADD_DECLARATION;

      ADD_STYLE_RULE;
   }
   /*QMenu::item {
       -qt-style-features: background-color;
   }*/
   if (!styleIsPixmapBased) {
      SET_ELEMENT_NAME(QLatin1String("QMenu"));
      ADD_PSEUDO(QLatin1String("item"), PseudoClass_Unknown);
      ADD_BASIC_SELECTOR;
      ADD_SELECTOR;

      SET_PROPERTY(QLatin1String("-qt-style-features"), QtStyleFeatures);
      ADD_VALUE(Value::Identifier, QString::fromLatin1("background-color"));
      ADD_DECLARATION;

      ADD_STYLE_RULE;
   }

   /*QHeaderView {
       -qt-background-role: window;
   }*/
   {
      SET_ELEMENT_NAME(QLatin1String("QHeaderView"));
      ADD_BASIC_SELECTOR;
      ADD_SELECTOR;

      SET_PROPERTY(QLatin1String("-qt-background-role"), QtBackgroundRole);
      ADD_VALUE(Value::KnownIdentifier, Value_Window);
      ADD_DECLARATION;

      ADD_STYLE_RULE;
   }

   /*QTableCornerButton::section, QHeaderView::section {
       -qt-background-role: button;
       -qt-style-features: background-color; //if style is not pixmap based
       border: native;
   }*/
   {
      SET_ELEMENT_NAME(QLatin1String("QTableCornerButton"));
      ADD_PSEUDO(QLatin1String("section"), PseudoClass_Unknown);
      ADD_BASIC_SELECTOR;
      ADD_SELECTOR;

      SET_ELEMENT_NAME(QLatin1String("QHeaderView"));
      ADD_PSEUDO(QLatin1String("section"), PseudoClass_Unknown);
      ADD_BASIC_SELECTOR;
      ADD_SELECTOR;

      SET_PROPERTY(QLatin1String("-qt-background-role"), QtBackgroundRole);
      ADD_VALUE(Value::KnownIdentifier, Value_Button);
      ADD_DECLARATION;

      if (!styleIsPixmapBased) {
         SET_PROPERTY(QLatin1String("-qt-style-features"), QtStyleFeatures);
         ADD_VALUE(Value::Identifier, QString::fromLatin1("background-color"));
         ADD_DECLARATION;
      }

      SET_PROPERTY(QLatin1String("border"), Border);
      ADD_VALUE(Value::KnownIdentifier, Value_Native);
      ADD_DECLARATION;

      ADD_STYLE_RULE;
   }

   /*QProgressBar {
       -qt-background-role: base;
   }*/
   {
      SET_ELEMENT_NAME(QLatin1String("QProgressBar"));
      ADD_BASIC_SELECTOR;
      ADD_SELECTOR;

      SET_PROPERTY(QLatin1String("-qt-background-role"), QtBackgroundRole);
      ADD_VALUE(Value::KnownIdentifier, Value_Base);
      ADD_DECLARATION;

      ADD_STYLE_RULE;
   }

   /*QScrollBar {
       -qt-background-role: window;
   }*/
   {
      SET_ELEMENT_NAME(QLatin1String("QScrollBar"));
      ADD_BASIC_SELECTOR;
      ADD_SELECTOR;

      SET_PROPERTY(QLatin1String("-qt-background-role"), QtBackgroundRole);
      ADD_VALUE(Value::KnownIdentifier, Value_Window);
      ADD_DECLARATION;

      ADD_STYLE_RULE;
   }

   /*QDockWidget {
       border: native;
   }*/
   {
      SET_ELEMENT_NAME(QLatin1String("QDockWidget"));
      ADD_BASIC_SELECTOR;
      ADD_SELECTOR;

      SET_PROPERTY(QLatin1String("border"), Border);
      ADD_VALUE(Value::KnownIdentifier, Value_Native);
      ADD_DECLARATION;

      ADD_STYLE_RULE;
   }

   sheet.origin = StyleSheetOrigin_UserAgent;
   sheet.buildIndexes();
   return sheet;
}

#endif // QT_NO_STYLE_STYLESHEET

