/***********************************************************************
*
* Copyright (c) 2012-2026 Barbara Geller
* Copyright (c) 2012-2026 Ansel Sermersheim
*
* Copyright (c) 2015 The Qt Company Ltd.
* Copyright (c) 2012-2016 Digia Plc and/or its subsidiary(-ies).
* Copyright (c) 2008-2012 Nokia Corporation and/or its subsidiary(-ies).
*
* This file is part of CopperSpice.
*
* CopperSpice is free software. You can redistribute it and/or
* modify it under the terms of the GNU Lesser General Public License
* version 2.1 as published by the Free Software Foundation.
*
* CopperSpice is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
*
* https://www.gnu.org/licenses/
*
***********************************************************************/

#ifndef QFRAGMENTMAP_P_H
#define QFRAGMENTMAP_P_H

#include <qglobal.h>

#include <qtools_p.h>

#include <stdlib.h>

template <int N = 1>
class QFragment
{
 public:
   static constexpr const int size_array_max = N;

   quint32 parent;
   quint32 left;
   quint32 right;
   quint32 color;
   quint32 size_left_array[N];
   quint32 size_array[N];
};

template <class Fragment>
class QFragmentMapData
{
 public:
   static constexpr const int fragmentSize = sizeof(Fragment);

   QFragmentMapData();
   ~QFragmentMapData();

   void init();

   class Header
   {
    public:
      quint32 root; // this relies on being at the same position as parent in the fragment struct
      quint32 tag;
      quint32 freelist;
      quint32 node_count;
      quint32 allocated;
   };

   int length(uint field = 0) const;

   Fragment *fragment(uint index) {
      return (fragments + index);
   }

   const Fragment *fragment(uint index) const {
      return (fragments + index);
   }

   Fragment &F(uint index) {
      return fragments[index] ;
   }

   const Fragment &F(uint index) const {
      return fragments[index] ;
   }

   bool isRoot(uint index) const {
      return !fragment(index)->parent;
   }

   uint position(uint node, uint field = 0) const {
      Q_ASSERT(field < Fragment::size_array_max);

      const Fragment *f = fragment(node);
      uint offset = f->size_left_array[field];

      while (f->parent) {
         uint p = f->parent;
         f = fragment(p);
         if (f->right == node) {
            offset += f->size_left_array[field] + f->size_array[field];
         }
         node = p;
      }
      return offset;
   }

   uint sizeRight(uint node, uint field = 0) const {
      Q_ASSERT(field < Fragment::size_array_max);
      uint sr = 0;
      const Fragment *f = fragment(node);
      node = f->right;

      while (node) {
         f = fragment(node);
         sr += f->size_left_array[field] + f->size_array[field];
         node = f->right;
      }
      return sr;
   }

   uint sizeLeft(uint node, uint field = 0) const {
      Q_ASSERT(field < Fragment::size_array_max);
      return fragment(node)->size_left_array[field];
   }

   uint size(uint node, uint field = 0) const {
      Q_ASSERT(field < Fragment::size_array_max);
      return fragment(node)->size_array[field];
   }

   void setSize(uint node, int new_size, uint field = 0) {
      Q_ASSERT(field < Fragment::size_array_max);
      Fragment *f = fragment(node);
      int diff = new_size - f->size_array[field];
      f->size_array[field] = new_size;
      while (f->parent) {
         uint p = f->parent;
         f = fragment(p);
         if (f->left == node) {
            f->size_left_array[field] += diff;
         }
         node = p;
      }
   }

   uint findNode(int k, uint field = 0) const;

   uint insert_single(int key, uint length);
   uint erase_single(uint f);

   uint minimum(uint n) const {
      while (n && fragment(n)->left) {
         n = fragment(n)->left;
      }
      return n;
   }

   uint maximum(uint n) const {
      while (n && fragment(n)->right) {
         n = fragment(n)->right;
      }
      return n;
   }

   uint next(uint n) const;
   uint previous(uint n) const;

   uint root() const {
      Q_ASSERT(!head->root || !fragment(head->root)->parent);
      return head->root;
   }

   void setRoot(uint new_root) {
      Q_ASSERT(!head->root || !fragment(new_root)->parent);
      head->root = new_root;
   }

   bool isValid(uint n) const {
      return n > 0 && n != head->freelist;
   }

   union {
      Header *head;
      Fragment *fragments;
   };

 private:
   enum Color {
      Red,
      Black
   };

   void rotateLeft(uint x);
   void rotateRight(uint x);
   void rebalance(uint x);
   void removeAndRebalance(uint z);

   uint createFragment();
   void freeFragment(uint f);
};

template <class Fragment>
QFragmentMapData<Fragment>::QFragmentMapData()
   : fragments(nullptr )
{
   init();
}

template <class Fragment>
void QFragmentMapData<Fragment>::init()
{
   // the following code will realloc an existing fragment or create a new one.
   // it will also ignore errors when shrinking an existing fragment.
   Fragment *newFragments = (Fragment *)realloc(fragments, 64 * fragmentSize);
   if (newFragments) {
      fragments = newFragments;
      head->allocated = 64;
   }

   Q_CHECK_PTR(fragments);

   head->tag = (((quint32)'p') << 24) | (((quint32)'m') << 16) | (((quint32)'a') << 8) | 'p'; //TAG('p', 'm', 'a', 'p');
   head->root = 0;
   head->freelist = 1;
   head->node_count = 0;

   // mark all items to the right as unused
   F(head->freelist).right = 0;
}

template <class Fragment>
QFragmentMapData<Fragment>::~QFragmentMapData()
{
   free(fragments);
}

template <class Fragment>
uint QFragmentMapData<Fragment>::createFragment()
{
   Q_ASSERT(head->freelist <= head->allocated);

   uint freePos = head->freelist;
   if (freePos == head->allocated) {
      // need to create some free space
      if (freePos >= uint(MaxAllocSize) / fragmentSize) {
         qBadAlloc();
      }

      uint needed = qAllocMore((freePos + 1) * fragmentSize, 0);
      Q_ASSERT(needed / fragmentSize > head->allocated);
      Fragment *newFragments = (Fragment *)realloc(fragments, needed);
      Q_CHECK_PTR(newFragments);
      fragments = newFragments;
      head->allocated = needed / fragmentSize;
      F(freePos).right = 0;
   }

   uint nextPos = F(freePos).right;
   if (!nextPos) {
      nextPos = freePos + 1;
      if (nextPos < head->allocated) {
         F(nextPos).right = 0;
      }
   }

   head->freelist = nextPos;

   ++head->node_count;

   return freePos;
}

template <class Fragment>
void QFragmentMapData<Fragment>::freeFragment(uint i)
{
   F(i).right = head->freelist;
   head->freelist = i;

   --head->node_count;
}

template <class Fragment>
uint QFragmentMapData<Fragment>::next(uint n) const
{
   Q_ASSERT(n);
   if (F(n).right) {
      n = F(n).right;
      while (F(n).left) {
         n = F(n).left;
      }
   } else {
      uint y = F(n).parent;
      while (F(n).parent && n == F(y).right) {
         n = y;
         y = F(y).parent;
      }
      n = y;
   }
   return n;
}

template <class Fragment>
uint QFragmentMapData<Fragment>::previous(uint n) const
{
   if (!n) {
      return maximum(root());
   }

   if (F(n).left) {
      n = F(n).left;
      while (F(n).right) {
         n = F(n).right;
      }
   } else {
      uint y = F(n).parent;
      while (F(n).parent && n == F(y).left) {
         n = y;
         y = F(y).parent;
      }
      n = y;
   }
   return n;
}


/*
     x              y
      \            / \
       y    -->   x   b
      / \          \
     a   b          a
*/
template <class Fragment>
void QFragmentMapData<Fragment>::rotateLeft(uint x)
{
   uint p = F(x).parent;
   uint y = F(x).right;


   if (y) {
      F(x).right = F(y).left;
      if (F(y).left) {
         F(F(y).left).parent = x;
      }
      F(y).left = x;
      F(y).parent = p;
   } else {
      F(x).right = 0;
   }

   if (!p) {
      Q_ASSERT(head->root == x);
      head->root = y;
   } else if (x == F(p).left) {
      F(p).left = y;
   } else {
      F(p).right = y;
   }

   F(x).parent = y;
   for (uint field = 0; field < Fragment::size_array_max; ++field) {
      F(y).size_left_array[field] += F(x).size_left_array[field] + F(x).size_array[field];
   }
}


/*
         x          y
        /          / \
       y    -->   a   x
      / \            /
     a   b          b
*/
template <class Fragment>
void QFragmentMapData<Fragment>::rotateRight(uint x)
{
   uint y = F(x).left;
   uint p = F(x).parent;

   if (y) {
      F(x).left = F(y).right;
      if (F(y).right) {
         F(F(y).right).parent = x;
      }
      F(y).right = x;
      F(y).parent = p;
   } else {
      F(x).left = 0;
   }
   if (!p) {
      Q_ASSERT(head->root == x);
      head->root = y;
   } else if (x == F(p).right) {
      F(p).right = y;
   } else {
      F(p).left = y;
   }
   F(x).parent = y;
   for (uint field = 0; field < Fragment::size_array_max; ++field) {
      F(x).size_left_array[field] -= F(y).size_left_array[field] + F(y).size_array[field];
   }
}


template <class Fragment>
void QFragmentMapData<Fragment>::rebalance(uint x)
{
   F(x).color = Red;

   while (F(x).parent && F(F(x).parent).color == Red) {
      uint p = F(x).parent;
      uint pp = F(p).parent;
      Q_ASSERT(pp);
      if (p == F(pp).left) {
         uint y = F(pp).right;
         if (y && F(y).color == Red) {
            F(p).color = Black;
            F(y).color = Black;
            F(pp).color = Red;
            x = pp;
         } else {
            if (x == F(p).right) {
               x = p;
               rotateLeft(x);
               p = F(x).parent;
               pp = F(p).parent;
            }
            F(p).color = Black;
            if (pp) {
               F(pp).color = Red;
               rotateRight(pp);
            }
         }
      } else {
         uint y = F(pp).left;
         if (y && F(y).color == Red) {
            F(p).color = Black;
            F(y).color = Black;
            F(pp).color = Red;
            x = pp;
         } else {
            if (x == F(p).left) {
               x = p;
               rotateRight(x);
               p = F(x).parent;
               pp = F(p).parent;
            }
            F(p).color = Black;
            if (pp) {
               F(pp).color = Red;
               rotateLeft(pp);
            }
         }
      }
   }
   F(root()).color = Black;
}

template <class Fragment>
uint QFragmentMapData<Fragment>::erase_single(uint z)
{
   uint w = previous(z);
   uint y = z;
   uint x;
   uint p;

   if (!F(y).left) {
      x = F(y).right;
   } else if (!F(y).right) {
      x = F(y).left;
   } else {
      y = F(y).right;
      while (F(y).left) {
         y = F(y).left;
      }
      x = F(y).right;
   }

   if (y != z) {
      F(F(z).left).parent = y;
      F(y).left = F(z).left;
      for (uint field = 0; field < Fragment::size_array_max; ++field) {
         F(y).size_left_array[field] = F(z).size_left_array[field];
      }
      if (y != F(z).right) {
         /*
                  z                y
                 / \              / \
                a   b            a   b
                   /                /
                 ...     -->      ...
                 /                /
                y                x
               / \
              0   x
          */
         p = F(y).parent;
         if (x) {
            F(x).parent = p;
         }
         F(p).left = x;
         F(y).right = F(z).right;
         F(F(z).right).parent = y;
         uint n = p;
         while (n != y) {
            for (uint field = 0; field < Fragment::size_array_max; ++field) {
               F(n).size_left_array[field] -= F(y).size_array[field];
            }
            n = F(n).parent;
         }
      } else {
         /*
                  z                y
                 / \              / \
                a   y     -->    a   x
                   / \
                  0   x
          */
         p = y;
      }
      uint zp = F(z).parent;
      if (!zp) {
         Q_ASSERT(head->root == z);
         head->root = y;
      } else if (F(zp).left == z) {
         F(zp).left = y;
         for (uint field = 0; field < Fragment::size_array_max; ++field) {
            F(zp).size_left_array[field] -= F(z).size_array[field];
         }
      } else {
         F(zp).right = y;
      }
      F(y).parent = zp;
      // Swap the colors
      uint c = F(y).color;
      F(y).color = F(z).color;
      F(z).color = c;
      y = z;
   } else {
      /*
              p          p            p          p
             /          /              \          \
            z    -->   x                z  -->     x
            |                           |
            x                           x
       */
      p = F(z).parent;
      if (x) {
         F(x).parent = p;
      }
      if (!p) {
         Q_ASSERT(head->root == z);
         head->root = x;
      } else if (F(p).left == z) {
         F(p).left = x;
         for (uint field = 0; field < Fragment::size_array_max; ++field) {
            F(p).size_left_array[field] -= F(z).size_array[field];
         }
      } else {
         F(p).right = x;
      }
   }
   uint n = z;
   while (F(n).parent) {
      uint p2 = F(n).parent;
      if (F(p2).left == n) {
         for (uint field = 0; field < Fragment::size_array_max; ++field) {
            F(p2).size_left_array[field] -= F(z).size_array[field];
         }
      }
      n = p2;
   }

   freeFragment(z);


   if (F(y).color != Red) {
      while (F(x).parent && (x == 0 || F(x).color == Black)) {
         if (x == F(p).left) {
            uint w2 = F(p).right;
            if (F(w2).color == Red) {
               F(w2).color = Black;
               F(p).color = Red;
               rotateLeft(p);
               w2 = F(p).right;
            }
            if ((F(w2).left == 0 || F(F(w2).left).color == Black) &&
               (F(w2).right == 0 || F(F(w2).right).color == Black)) {
               F(w2).color = Red;
               x = p;
               p = F(x).parent;
            } else {
               if (F(w2).right == 0 || F(F(w2).right).color == Black) {
                  if (F(w2).left) {
                     F(F(w2).left).color = Black;
                  }
                  F(w2).color = Red;
                  rotateRight(F(p).right);
                  w2 = F(p).right;
               }
               F(w2).color = F(p).color;
               F(p).color = Black;
               if (F(w2).right) {
                  F(F(w2).right).color = Black;
               }
               rotateLeft(p);
               break;
            }
         } else {
            uint w3 = F(p).left;
            if (F(w3).color == Red) {
               F(w3).color = Black;
               F(p).color = Red;
               rotateRight(p);
               w3 = F(p).left;
            }
            if ((F(w3).right == 0 || F(F(w3).right).color == Black) &&
               (F(w3).left == 0 || F(F(w3).left).color == Black)) {
               F(w3).color = Red;
               x = p;
               p = F(x).parent;
            } else {
               if (F(w3).left == 0 || F(F(w3).left).color == Black) {
                  if (F(w3).right) {
                     F(F(w3).right).color = Black;
                  }
                  F(w3).color = Red;
                  rotateLeft(F(p).left);
                  w3 = F(p).left;
               }
               F(w3).color = F(p).color;
               F(p).color = Black;
               if (F(w3).left) {
                  F(F(w3).left).color = Black;
               }
               rotateRight(p);
               break;
            }
         }
      }
      if (x) {
         F(x).color = Black;
      }
   }

   return w;
}

template <class Fragment>
uint QFragmentMapData<Fragment>::findNode(int k, uint field) const
{
   Q_ASSERT(field < Fragment::size_array_max);
   uint x = root();

   uint s = k;
   while (x) {
      if (sizeLeft(x, field) <= s) {
         if (s < sizeLeft(x, field) + size(x, field)) {
            return x;
         }
         s -= sizeLeft(x, field) + size(x, field);
         x = F(x).right;
      } else {
         x = F(x).left;
      }
   }
   return 0;
}

template <class Fragment>
uint QFragmentMapData<Fragment>::insert_single(int key, uint length)
{
   Q_ASSERT(!findNode(key) || (int)this->position(findNode(key)) == key);

   uint z = createFragment();

   F(z).left = 0;
   F(z).right = 0;
   F(z).size_array[0] = length;
   for (uint field = 1; field < Fragment::size_array_max; ++field) {
      F(z).size_array[field] = 1;
   }
   for (uint field = 0; field < Fragment::size_array_max; ++field) {
      F(z).size_left_array[field] = 0;
   }

   uint y = 0;
   uint x = root();

   Q_ASSERT(!x || F(x).parent == 0);

   uint s = key;
   bool right = false;
   while (x) {
      y = x;
      if (s <= F(x).size_left_array[0]) {
         x = F(x).left;
         right = false;
      } else {
         s -= F(x).size_left_array[0] + F(x).size_array[0];
         x = F(x).right;
         right = true;
      }
   }

   F(z).parent = y;
   if (!y) {
      head->root = z;
   } else if (!right) {
      F(y).left = z;
      for (uint field = 0; field < Fragment::size_array_max; ++field) {
         F(y).size_left_array[field] = F(z).size_array[field];
      }
   } else {
      F(y).right = z;
   }
   while (y && F(y).parent) {
      uint p = F(y).parent;
      if (F(p).left == y) {
         for (uint field = 0; field < Fragment::size_array_max; ++field) {
            F(p).size_left_array[field] += F(z).size_array[field];
         }
      }
      y = p;
   }
   rebalance(z);

   return z;
}

template <class Fragment>
int QFragmentMapData<Fragment>::length(uint field) const
{
   uint root = this->root();
   return root ? sizeLeft(root, field) + size(root, field) + sizeRight(root, field) : 0;
}


template <class Fragment>       // NOTE: must inherit QFragment
class QFragmentMap
{
 public:
   class iterator
   {
    public:
      iterator() : pt(0), n(0) {}
      iterator(QFragmentMap *p, int node) : pt(p), n(node) {}
      iterator(const iterator &it) : pt(it.pt), n(it.n) {}

      bool atEnd() const {
         return !n;
      }

      bool operator==(const iterator &it) const {
         return pt == it.pt && n == it.n;
      }

      bool operator!=(const iterator &it) const {
         return pt != it.pt || n != it.n;
      }

      bool operator<(const iterator &it) const {
         return position() < it.position();
      }

      Fragment *operator*() {
         Q_ASSERT(!atEnd());
         return pt->fragment(n);
      }
      const Fragment *operator*() const {
         Q_ASSERT(!atEnd());
         return pt->fragment(n);
      }
      Fragment *operator->() {
         Q_ASSERT(!atEnd());
         return pt->fragment(n);
      }
      const Fragment *operator->() const {
         Q_ASSERT(!atEnd());
         return pt->fragment(n);
      }

      int position() const {
         Q_ASSERT(!atEnd());
         return pt->data.position(n);
      }
      const Fragment *value() const {
         Q_ASSERT(!atEnd());
         return pt->fragment(n);
      }
      Fragment *value() {
         Q_ASSERT(!atEnd());
         return pt->fragment(n);
      }

      iterator &operator++() {
         n = pt->data.next(n);
         return *this;
      }
      iterator &operator--() {
         n = pt->data.previous(n);
         return *this;
      }

      QFragmentMap *pt;
      quint32 n;
   };


   class const_iterator
   {
    public:
      const_iterator() : pt(0), n(0) {}
      const_iterator(const QFragmentMap *p, int node) : pt(p), n(node) {}
      const_iterator(const const_iterator &it) : pt(it.pt), n(it.n) {}
      const_iterator(const iterator &it) : pt(it.pt), n(it.n) {}

      bool atEnd() const {
         return !n;
      }

      bool operator==(const const_iterator &it) const {
         return pt == it.pt && n == it.n;
      }
      bool operator!=(const const_iterator &it) const {
         return pt != it.pt || n != it.n;
      }
      bool operator<(const const_iterator &it) const {
         return position() < it.position();
      }

      const Fragment *operator*()  const {
         Q_ASSERT(!atEnd());
         return pt->fragment(n);
      }
      const Fragment *operator->()  const {
         Q_ASSERT(!atEnd());
         return pt->fragment(n);
      }

      int position() const {
         Q_ASSERT(!atEnd());
         return pt->data.position(n);
      }
      int size() const {
         Q_ASSERT(!atEnd());
         return pt->data.size(n);
      }
      const Fragment *value() const {
         Q_ASSERT(!atEnd());
         return pt->fragment(n);
      }

      const_iterator &operator++() {
         n = pt->data.next(n);
         return *this;
      }
      const_iterator &operator--() {
         n = pt->data.previous(n);
         return *this;
      }

      const QFragmentMap *pt;
      quint32 n;
   };


   QFragmentMap()
   { }

   ~QFragmentMap() {
      if (!data.fragments) {
         return;   // in case of out-of-memory, we won't have fragments
      }
      for (iterator it = begin(); !it.atEnd(); ++it) {
         it.value()->free();
      }
   }

   void clear() {
      for (iterator it = begin(); !it.atEnd(); ++it) {
         it.value()->free();
      }
      data.init();
   }

   iterator begin() {
      return iterator(this, data.minimum(data.root()));
   }

   iterator end() {
      return iterator(this, 0);
   }

   const_iterator begin() const {
      return const_iterator(this, data.minimum(data.root()));
   }

   const_iterator end() const {
      return const_iterator(this, 0);
   }

   const_iterator last() const {
      return const_iterator(this, data.maximum(data.root()));
   }

   bool isEmpty() const {
      return data.head->node_count == 0;
   }

   int numNodes() const {
      return data.head->node_count;
   }

   int length(uint field = 0) const {
      return data.length(field);
   }

   iterator find(int k, uint field = 0) {
      return iterator(this, data.findNode(k, field));
   }
   const_iterator find(int k, uint field = 0) const {
      return const_iterator(this, data.findNode(k, field));
   }

   uint findNode(int k, uint field = 0) const {
      return data.findNode(k, field);
   }

   uint insert_single(int key, uint length) {
      uint f = data.insert_single(key, length);
      if (f != 0) {
         Fragment *frag = fragment(f);
         Q_ASSERT(frag);
         frag->initialize();
      }
      return f;
   }

   uint erase_single(uint f) {
      if (f != 0) {
         Fragment *frag = fragment(f);
         Q_ASSERT(frag);
         frag->free();
      }
      return data.erase_single(f);
   }

   Fragment *fragment(uint index) {
      Q_ASSERT(index != 0);
      return data.fragment(index);
   }

   const Fragment *fragment(uint index) const {
      Q_ASSERT(index != 0);
      return data.fragment(index);
   }

   uint position(uint node, uint field = 0) const {
      return data.position(node, field);
   }

   bool isValid(uint n) const {
      return data.isValid(n);
   }

   uint next(uint n) const {
      return data.next(n);
   }

   uint previous(uint n) const {
      return data.previous(n);
   }

   uint size(uint node, uint field = 0) const {
      return data.size(node, field);
   }

   void setSize(uint node, int new_size, uint field = 0) {
      data.setSize(node, new_size, field);

      if (node != 0 && field == 0) {
         Fragment *frag = fragment(node);
         Q_ASSERT(frag);
         frag->invalidate();
      }
   }

   int firstNode() const {
      return data.minimum(data.root());
   }

 private:
   friend class iterator;
   friend class const_iterator;

   QFragmentMapData<Fragment> data;

   QFragmentMap(const QFragmentMap &m);
   QFragmentMap &operator= (const QFragmentMap &m);
};

#endif
