/***********************************************************************
*
* Copyright (c) 2012-2026 Barbara Geller
* Copyright (c) 2012-2026 Ansel Sermersheim
*
* Copyright (c) 2015 The Qt Company Ltd.
* Copyright (c) 2012-2016 Digia Plc and/or its subsidiary(-ies).
* Copyright (c) 2008-2012 Nokia Corporation and/or its subsidiary(-ies).
*
* This file is part of CopperSpice.
*
* CopperSpice is free software. You can redistribute it and/or
* modify it under the terms of the GNU Lesser General Public License
* version 2.1 as published by the Free Software Foundation.
*
* CopperSpice is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
*
* https://www.gnu.org/licenses/
*
***********************************************************************/

#include <qsslcontext_openssl_p.h>

#include <qmutex.h>
#include <qsslsocket.h>
#include <qstring.h>

#include <qsslsocket_openssl_p.h>
#include <qsslsocket_openssl_symbols_p.h>
#include <qsslsocket_p.h>

// defined in qsslsocket_openssl.cpp:
extern int q_X509Callback(int ok, X509_STORE_CTX *ctx);
extern QString getErrorsFromOpenSsl();

static DH *get_dh1024()
{
   // Default DH params
   // 1024-bit MODP Group
   // From RFC 2409
   QByteArray params = QByteArray::fromBase64(QByteArrayLiteral("MIGHAoGBAP//////////yQ/aoiFowjTExmKLgNwc0SkCTgiKZ8x0Agu+pjsTmyJR" \
                       "Sgh5jjQE3e+VGbPNOkMbMCsKbfJfFDdP4TVtbVHCReSFtXZiXn7G9ExC6aY37WsL" \
                       "/1y29Aa37e44a/taiZ+lrp8kEXxLH+ZJKGZR7OZTgf//////////AgEC"));

   const char *ptr = params.constData();
   DH *dh = q_d2i_DHparams(nullptr, reinterpret_cast<const unsigned char **>(&ptr), params.length());

   return dh;
}

QSslContext::QSslContext()
   : ctx(nullptr), pkey(nullptr), session(nullptr), m_sessionTicketLifeTimeHint(-1)
{
}

QSslContext::~QSslContext()
{
   if (ctx) {
      //  will decrement the reference count by 1 and free the context eventually when possible
      q_SSL_CTX_free(ctx);
   }

   if (pkey) {
      q_EVP_PKEY_free(pkey);
   }

   if (session) {
      q_SSL_SESSION_free(session);
   }
}

static inline QString msgErrorSettingEllipticCurves(const QString &why)
{
   return QSslSocket::tr("Error when setting the elliptic curves (%1)").formatArg(why);
}

QSslContext *QSslContext::fromConfiguration(QSslSocket::SslMode mode,
      const QSslConfiguration &configuration, bool allowRootCertOnDemandLoading)
{
   QSslContext *sslContext = new QSslContext();
   sslContext->sslConfiguration = configuration;
   sslContext->errorCode = QSslError::NoError;

   bool client = (mode == QSslSocket::SslClientMode);

   bool reinitialized = false;
   bool unsupportedProtocol = false;

init_context:
   switch (sslContext->sslConfiguration.protocol()) {

      case QSsl::SslV2:
#ifndef OPENSSL_NO_SSL2

#if OPENSSL_VERSION_NUMBER < 0x10100000L
         sslContext->ctx = q_SSL_CTX_new(client ? q_SSLv2_client_method() : q_SSLv2_server_method());
#elif OPENSSL_VERSION_NUMBER >= 0x10100000L
         sslContext->ctx = q_SSL_CTX_new(client ? q_TLS_client_method() : q_TLS_server_method());
#endif

#else
         // SSL 2 not supported by the system, but chosen deliberately -> error
         sslContext->ctx = nullptr;
         unsupportedProtocol = true;
#endif
         break;

      case QSsl::SslV3:
#ifndef OPENSSL_NO_SSL3_METHOD

#if OPENSSL_VERSION_NUMBER < 0x10100000L
         sslContext->ctx = q_SSL_CTX_new(client ? q_SSLv3_client_method() : q_SSLv3_server_method());
#elif OPENSSL_VERSION_NUMBER >= 0x10100000L
         sslContext->ctx = q_SSL_CTX_new(client ? q_TLS_client_method() : q_TLS_server_method());
#endif

#else
         // SSL 3 not supported by the system, but chosen deliberately -> error
         sslContext->ctx = nullptr;
         unsupportedProtocol = true;
#endif
         break;

      case QSsl::SecureProtocols:

      // SSLv2 and SSLv3 will be disabled by SSL options
      // But we need q_SSLv23_server_method() otherwise AnyProtocol will be unable to connect on Win32.
      case QSsl::TlsV1SslV3:

      // SSLv2 will will be disabled by SSL options
      case QSsl::AnyProtocol:

      default:
#if OPENSSL_VERSION_NUMBER < 0x10100000L
         sslContext->ctx = q_SSL_CTX_new(client ? q_SSLv23_client_method() : q_SSLv23_server_method());
#else
         sslContext->ctx = q_SSL_CTX_new(client ? q_TLS_client_method() : q_TLS_server_method());
#endif
         break;

      case QSsl::TlsV1_0:
         sslContext->ctx = q_SSL_CTX_new(client ? q_TLSv1_client_method() : q_TLSv1_server_method());
         break;

      case QSsl::TlsV1_1:
#if OPENSSL_VERSION_NUMBER >= 0x10001000L
         sslContext->ctx = q_SSL_CTX_new(client ? q_TLSv1_1_client_method() : q_TLSv1_1_server_method());
#else
         // TLS 1.1 not supported by the system, but chosen deliberately -> error
         sslContext->ctx = 0;
         unsupportedProtocol = true;
#endif
         break;

      case QSsl::TlsV1_2:
#if OPENSSL_VERSION_NUMBER >= 0x10001000L
         sslContext->ctx = q_SSL_CTX_new(client ? q_TLSv1_2_client_method() : q_TLSv1_2_server_method());
#else
         // TLS 1.2 not supported by the system, but chosen deliberately -> error
         sslContext->ctx = 0;
         unsupportedProtocol = true;
#endif
         break;

      case QSsl::TlsV1_0_OrLater:
#if OPENSSL_VERSION_NUMBER < 0x10100000L
         // Specific protocols will be specified via SSL options.
         sslContext->ctx = q_SSL_CTX_new(client ? q_SSLv23_client_method() : q_SSLv23_server_method());
#else
         sslContext->ctx = q_SSL_CTX_new(client ? q_TLS_client_method() : q_TLS_server_method());
#endif
         break;

      case QSsl::TlsV1_1_OrLater:
      case QSsl::TlsV1_2_OrLater:
#if OPENSSL_VERSION_NUMBER >= 0x10001000L && OPENSSL_VERSION_NUMBER < 0x10100000L
         // Specific protocols will be specified via SSL options.
         sslContext->ctx = q_SSL_CTX_new(client ? q_SSLv23_client_method() : q_SSLv23_server_method());
#elif OPENSSL_VERSION_NUMBER >= 0x10100000L
         sslContext->ctx = q_SSL_CTX_new(client ? q_TLS_client_method() : q_TLS_server_method());
#else
         // TLS 1.1/1.2 not supported by the system, but chosen deliberately -> error
         sslContext->ctx = 0;
         unsupportedProtocol = true;
#endif
         break;
   }

   if (! sslContext->ctx) {
      // After stopping Flash 10 the SSL library looses its ciphers. Try re-adding them
      // by re-initializing the library.
      if (!reinitialized) {
         reinitialized = true;

#if OPENSSL_VERSION_NUMBER < 0x10100000L
         if (q_SSL_library_init() == 1) {
#else
         if (q_OPENSSL_init_ssl(0, nullptr) == 1 && q_OPENSSL_init_crypto(0, nullptr) == 1) {
#endif
            goto init_context;
         }
      }

      sslContext->errorStr = QSslSocket::tr("Error creating SSL context (%1)")
                  .formatArg(unsupportedProtocol ? QSslSocket::tr("unsupported protocol") : QSslSocketBackendPrivate::getErrorsFromOpenSsl());

      sslContext->errorCode = QSslError::UnspecifiedError;
      return sslContext;
   }

   // Enable bug workarounds.
   long options = QSslSocketBackendPrivate::setupOpenSslOptions(configuration.protocol(), configuration.d->sslOptions);
   q_SSL_CTX_set_options(sslContext->ctx, options);

#if OPENSSL_VERSION_NUMBER >= 0x10000000L
   // Tell OpenSSL to release memory early
   // http://www.openssl.org/docs/ssl/SSL_CTX_set_mode.html
   if (q_SSLeay() >= 0x10000000L) {
      q_SSL_CTX_set_mode(sslContext->ctx, SSL_MODE_RELEASE_BUFFERS);
   }
#endif

   // Initialize ciphers
   QByteArray cipherString;
   bool first = true;
   QList<QSslCipher> ciphers = sslContext->sslConfiguration.ciphers();

   if (ciphers.isEmpty()) {
      ciphers = QSslSocketPrivate::defaultCiphers();
   }

   for (const QSslCipher &cipher : ciphers) {
      if (first) {
         first = false;
      } else {
         cipherString.append(':');
      }

      cipherString.append(cipher.name().toLatin1());
   }

   if (! q_SSL_CTX_set_cipher_list(sslContext->ctx, cipherString.data())) {
      sslContext->errorStr = QSslSocket::tr("Invalid or empty cipher list (%1)").formatArg(QSslSocketBackendPrivate::getErrorsFromOpenSsl());
      sslContext->errorCode = QSslError::UnspecifiedError;
      return sslContext;
   }

   const QDateTime now = QDateTime::currentDateTimeUtc();

   // Add all our CAs to this store.
   for (const QSslCertificate &caCertificate : sslContext->sslConfiguration.caCertificates()) {
      // From https://www.openssl.org/docs/ssl/SSL_CTX_load_verify_locations.html:
      //
      // If several CA certificates matching the name, key identifier, and
      // serial number condition are available, only the first one will be
      // examined. This may lead to unexpected results if the same CA
      // certificate is available with different expiration dates. If a
      // ``certificate expired'' verification error occurs, no other
      // certificate will be searched. Make sure to not have expired
      // certificates mixed with valid ones.
      //
      // See also: QSslSocketBackendPrivate::verify()

      if (caCertificate.expiryDate() >= now) {
         q_X509_STORE_add_cert(q_SSL_CTX_get_cert_store(sslContext->ctx), (X509 *)caCertificate.handle());
      }
   }

   if (QSslSocketPrivate::s_loadRootCertsOnDemand && allowRootCertOnDemandLoading) {
      // tell OpenSSL the directories where to look up the root certs on demand
      QList<QByteArray> unixDirs = QSslSocketPrivate::unixRootCertDirectories();

      for (int a = 0; a < unixDirs.count(); ++a) {
         q_SSL_CTX_load_verify_locations(sslContext->ctx, nullptr, unixDirs.at(a).constData());
      }
   }

   if (!sslContext->sslConfiguration.localCertificate().isNull()) {
      // Require a private key as well.
      if (sslContext->sslConfiguration.privateKey().isNull()) {
         sslContext->errorStr = QSslSocket::tr("Cannot provide a certificate with no key, %1").formatArg(QSslSocketBackendPrivate::getErrorsFromOpenSsl());
         sslContext->errorCode = QSslError::UnspecifiedError;
         return sslContext;
      }

      // Load certificate
      if (! q_SSL_CTX_use_certificate(sslContext->ctx, (X509 *)sslContext->sslConfiguration.localCertificate().handle())) {
         sslContext->errorStr = QSslSocket::tr("Error loading local certificate, %1").formatArg(QSslSocketBackendPrivate::getErrorsFromOpenSsl());
         sslContext->errorCode = QSslError::UnspecifiedError;
         return sslContext;
      }

      if (configuration.d->privateKey.algorithm() == QSsl::Opaque) {
         sslContext->pkey = reinterpret_cast<EVP_PKEY *>(configuration.d->privateKey.handle());

      } else {
         // Load private key
         sslContext->pkey = q_EVP_PKEY_new();

         // before we were using EVP_PKEY_assign_R* functions and did not use EVP_PKEY_free.
         // this lead to a memory leak. Now we use the *_set1_* functions which do not
         // take ownership of the RSA/DSA key instance because the QSslKey already has ownership.

         if (configuration.d->privateKey.algorithm() == QSsl::Rsa) {
            q_EVP_PKEY_set1_RSA(sslContext->pkey, reinterpret_cast<RSA *>(configuration.d->privateKey.handle()));

         } else if (configuration.d->privateKey.algorithm() == QSsl::Dsa) {
            q_EVP_PKEY_set1_DSA(sslContext->pkey, reinterpret_cast<DSA *>(configuration.d->privateKey.handle()));

#ifndef OPENSSL_NO_EC
         } else if (configuration.d->privateKey.algorithm() == QSsl::Ec) {
            q_EVP_PKEY_set1_EC_KEY(sslContext->pkey, reinterpret_cast<EC_KEY *>(configuration.d->privateKey.handle()));

#endif
         }
      }

      if (! q_SSL_CTX_use_PrivateKey(sslContext->ctx, sslContext->pkey)) {
         sslContext->errorStr = QSslSocket::tr("Error loading private key, %1").formatArg(QSslSocketBackendPrivate::getErrorsFromOpenSsl());
         sslContext->errorCode = QSslError::UnspecifiedError;
         return sslContext;
      }

      if (configuration.d->privateKey.algorithm() == QSsl::Opaque) {
         sslContext->pkey = nullptr;    // do not free the private key, it belongs to QSslKey
      }

      // Check if the certificate matches the private key.
      if (! q_SSL_CTX_check_private_key(sslContext->ctx)) {
         sslContext->errorStr = QSslSocket::tr("Private key does not certify public key, %1")
               .formatArg(QSslSocketBackendPrivate::getErrorsFromOpenSsl());

         sslContext->errorCode = QSslError::UnspecifiedError;
         return sslContext;
      }

      // If we have any intermediate certificates then we need to add them to our chain
      bool isFirst = true;

      for (const QSslCertificate &cert : configuration.d->localCertificateChain) {
         if (isFirst) {
            isFirst = false;
            continue;
         }

         q_SSL_CTX_ctrl(sslContext->ctx, SSL_CTRL_EXTRA_CHAIN_CERT, 0, q_X509_dup(reinterpret_cast<X509 *>(cert.handle())));
      }
   }

   // Initialize peer verification.
   if (sslContext->sslConfiguration.peerVerifyMode() == QSslSocket::VerifyNone) {
      q_SSL_CTX_set_verify(sslContext->ctx, SSL_VERIFY_NONE, nullptr);
   } else {
      q_SSL_CTX_set_verify(sslContext->ctx, SSL_VERIFY_PEER, q_X509Callback);
   }

   // Set verification depth.
   if (sslContext->sslConfiguration.peerVerifyDepth() != 0) {
      q_SSL_CTX_set_verify_depth(sslContext->ctx, sslContext->sslConfiguration.peerVerifyDepth());
   }

   // set persisted session if the user set it
   if (! configuration.sessionTicket().isEmpty()) {
      sslContext->setSessionASN1(configuration.sessionTicket());
   }

   // Set temp DH params
   DH *dh = nullptr;

   dh = get_dh1024();
   q_SSL_CTX_set_tmp_dh(sslContext->ctx, dh);
   q_DH_free(dh);

#ifndef OPENSSL_NO_EC

// ECDH is enabled by default since 1.1.0: https://github.com/openssl/openssl/issues/1437
#if OPENSSL_VERSION_NUMBER >= 0x10002000L && OPENSSL_VERSION_NUMBER <= 0x10100000L
   if (q_SSLeay() >= 0x10002000L) {
      q_SSL_CTX_ctrl(sslContext->ctx, SSL_CTRL_SET_ECDH_AUTO, 1, nullptr);

   } else

#endif
   {
      // Set temp ECDH params
      EC_KEY *ecdh = nullptr;

      ecdh = q_EC_KEY_new_by_curve_name(NID_X9_62_prime256v1);
      q_SSL_CTX_set_tmp_ecdh(sslContext->ctx, ecdh);
      q_EC_KEY_free(ecdh);
   }

#endif // OPENSSL_NO_EC

   const QVector<QSslEllipticCurve> qcurves = sslContext->sslConfiguration.ellipticCurves();

   if (!qcurves.isEmpty()) {

#if OPENSSL_VERSION_NUMBER >= 0x10002000L && !defined(OPENSSL_NO_EC)
      // Set the curves to be used
      if (q_SSLeay() >= 0x10002000L) {
         // SSL_CTX_ctrl wants a non-const pointer as last argument,
         // but let's avoid a copy into a temporary array

         if (! q_SSL_CTX_ctrl(sslContext->ctx, SSL_CTRL_SET_CURVES, qcurves.size(),
                              const_cast<int *>(reinterpret_cast<const int *>(qcurves.data())))) {

            sslContext->errorStr  = msgErrorSettingEllipticCurves(QSslSocketBackendPrivate::getErrorsFromOpenSsl());
            sslContext->errorCode = QSslError::UnspecifiedError;
            return sslContext;
         }

      } else

#endif
      {
         // specific curves requested, but not possible to set -> error
         sslContext->errorStr = msgErrorSettingEllipticCurves(QSslSocket::tr("OpenSSL version too old, need at least v1.0.2"));
         sslContext->errorCode = QSslError::UnspecifiedError;
         return sslContext;
      }
   }

   return sslContext;
}

#if OPENSSL_VERSION_NUMBER >= 0x1000100fL && ! defined(OPENSSL_NO_NEXTPROTONEG)

static int next_proto_cb(SSL *, unsigned char **out, unsigned char *outlen, const unsigned char *in, unsigned int inlen, void *arg)
{
   QSslContext::NPNContext *ctx = reinterpret_cast<QSslContext::NPNContext *>(arg);

   // comment out to debug:
   //    QList<QByteArray> supportedVersions;
   //    for (unsigned int i = 0; i < inlen; ) {
   //        QByteArray version(reinterpret_cast<const char *>(&in[i+1]), in[i]);
   //        supportedVersions << version;
   //        i += in[i] + 1;
   //    }

   int proto = q_SSL_select_next_proto(out, outlen, in, inlen, ctx->data, ctx->len);

   switch (proto) {
      case OPENSSL_NPN_UNSUPPORTED:
         ctx->status = QSslConfiguration::NextProtocolNegotiationNone;
         break;

      case OPENSSL_NPN_NEGOTIATED:
         ctx->status = QSslConfiguration::NextProtocolNegotiationNegotiated;
         break;

      case OPENSSL_NPN_NO_OVERLAP:
         ctx->status = QSslConfiguration::NextProtocolNegotiationUnsupported;
         break;

      default:
         qWarning("next_proto_cb() OpenSSL sent unknown NPN status");
   }

   return SSL_TLSEXT_ERR_OK;
}

QSslContext::NPNContext QSslContext::npnContext() const
{
   return m_npnContext;
}
#endif

// Needs to be deleted by caller
SSL *QSslContext::createSsl()
{
   SSL *ssl = q_SSL_new(ctx);
   q_SSL_clear(ssl);

   if (!session && !sessionASN1().isEmpty() && !sslConfiguration.testSslOption(QSsl::SslOptionDisableSessionPersistence)) {
      const unsigned char *data = reinterpret_cast<const unsigned char *>(m_sessionASN1.constData());
      session = q_d2i_SSL_SESSION(nullptr, &data, m_sessionASN1.size());   // refcount is 1 already, set
      // by function above
   }

   if (session) {
      // Try to resume the last session we cached
      if (! q_SSL_set_session(ssl, session)) {
         qWarning("QSslContext::createSsl() Unable to set SSL session");
         q_SSL_SESSION_free(session);
         session = nullptr;
      }
   }

#if OPENSSL_VERSION_NUMBER >= 0x1000100fL && !defined(OPENSSL_NO_NEXTPROTONEG)
   QList<QByteArray> protocols = sslConfiguration.d->nextAllowedProtocols;

   if (! protocols.isEmpty()) {
      m_supportedNPNVersions.clear();

      for (int a = 0; a < protocols.count(); ++a) {
         if (protocols.at(a).size() > 255) {
            qWarning()  << "QSslContext::createSsl() TLS NPN extension" << protocols.at(a)
                        << "is too long and will be truncated to 255 characters";

            protocols[a] = protocols.at(a).left(255);
         }

         m_supportedNPNVersions.append(protocols.at(a).size()).append(protocols.at(a));
      }

      m_npnContext.data = reinterpret_cast<unsigned char *>(m_supportedNPNVersions.data());
      m_npnContext.len = m_supportedNPNVersions.count();
      m_npnContext.status = QSslConfiguration::NextProtocolNegotiationNone;
      q_SSL_CTX_set_next_proto_select_cb(ctx, next_proto_cb, &m_npnContext);
   }

#endif // OPENSSL_VERSION_NUMBER >= 0x1000100fL ...

   return ssl;
}

// cache exactly one session here
bool QSslContext::cacheSession(SSL *ssl)
{
   // don't cache the same session again
   if (session && session == q_SSL_get_session(ssl)) {
      return true;
   }

   // decrease refcount of currently stored session
   // (this might happen if there are several concurrent handshakes in flight)
   if (session) {
      q_SSL_SESSION_free(session);
   }

   // cache the session the caller gave us and increase reference count
   session = q_SSL_get1_session(ssl);

   if (session && ! sslConfiguration.testSslOption(QSsl::SslOptionDisableSessionPersistence)) {
      int sessionSize = q_i2d_SSL_SESSION(session, nullptr);

      if (sessionSize > 0) {
         m_sessionASN1.resize(sessionSize);
         unsigned char *data = reinterpret_cast<unsigned char *>(m_sessionASN1.data());

         if (! q_i2d_SSL_SESSION(session, &data)) {
            qWarning("QSslContext::cacheSession() Unable to store persistent version of SSL session");
         }

#if OPENSSL_VERSION_NUMBER >= 0x10100000L
         m_sessionTicketLifeTimeHint = q_SSL_SESSION_get_ticket_lifetime_hint(session);
#else
         m_sessionTicketLifeTimeHint = session->tlsext_tick_lifetime_hint;
#endif
      }
   }

   return (session != nullptr);
}

QByteArray QSslContext::sessionASN1() const
{
   return m_sessionASN1;
}

void QSslContext::setSessionASN1(const QByteArray &sslSession)
{
   m_sessionASN1 = sslSession;
}

int QSslContext::sessionTicketLifeTimeHint() const
{
   return m_sessionTicketLifeTimeHint;
}

QSslError::SslError QSslContext::error() const
{
   return errorCode;
}

QString QSslContext::errorString() const
{
   return errorStr;
}
