/***********************************************************************
*
* Copyright (c) 2012-2026 Barbara Geller
* Copyright (c) 2012-2026 Ansel Sermersheim
*
* This file is part of CopperSpice.
*
* CopperSpice is free software. You can redistribute it and/or
* modify it under the terms of the GNU Lesser General Public License
* version 2.1 as published by the Free Software Foundation.
*
* CopperSpice is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
*
* https://www.gnu.org/licenses/
*
***********************************************************************/

#include <qdir.h>
#include <qtemporaryfile.h>
#include <qregularexpression.h>

#include <cs_catch2.h>

TEST_CASE("QTemporaryFile traits", "[qtemporaryfile]")
{
   REQUIRE(std::is_copy_constructible_v<QTemporaryFile> == false);
   REQUIRE(std::is_move_constructible_v<QTemporaryFile> == false);

   REQUIRE(std::is_copy_assignable_v<QTemporaryFile> == false);
   REQUIRE(std::is_move_assignable_v<QTemporaryFile> == false);

   REQUIRE(std::has_virtual_destructor_v<QTemporaryFile> == true);
}

TEST_CASE("QTemporaryFile fname_a", "[qtemporaryfile]")
{
   QTemporaryFile tmpFile;
   tmpFile.open();

   REQUIRE(tmpFile.fileName().contains("cs_temp.") == true);
   REQUIRE(tmpFile.fileName().endsWith(".XXXXXX") == false);
}

TEST_CASE("QTemporaryFile fname_b", "[qtemporaryfile]")
{
   QCoreApplication::setApplicationName("CsCoreTest");

   QTemporaryFile tmpFile;
   tmpFile.open();

   REQUIRE(tmpFile.fileName().contains("CsCoreTest") == true);
   REQUIRE(tmpFile.fileName().endsWith("XXXXXX") == false);

   QCoreApplication::setApplicationName("");
}

TEST_CASE("QTemporaryFile fname_c", "[qtemporaryfile]")
{
   QCoreApplication::setApplicationName("CsCoreTestXXXXXX.exe");

   QTemporaryFile tmpFile;
   tmpFile.open();

   REQUIRE(tmpFile.fileName().contains("CsCoreTestXXXXXX") == true);
   REQUIRE(tmpFile.fileName().endsWith("XXXXXX") == false);

   QCoreApplication::setApplicationName("");
}

TEST_CASE("QTemporaryFile fname_d", "[qtemporaryfile]")
{
   QCoreApplication::setApplicationName("CsCoreTest.exeXX");

   QTemporaryFile tmpFile;
   tmpFile.open();

   REQUIRE(tmpFile.fileName().contains("CsCoreTest") == true);
   REQUIRE(tmpFile.fileName().endsWith("XXXXXX") == false);

   QCoreApplication::setApplicationName("");
}

TEST_CASE("QTemporaryFile fname_e", "[qtemporaryfile]")
{
   QTemporaryFile tmpFile(QDir::tempPath() + "/MyCsCoreTestXXXXXX.name");
   tmpFile.open();

   REQUIRE(tmpFile.fileName().contains("MyCsCoreTest") == true);
   REQUIRE(tmpFile.fileName().contains("XXXXXX") == false);
   REQUIRE(tmpFile.fileName().endsWith(".name") == true);
}

TEST_CASE("QTemporaryFile create_fname_a", "[qtemporaryfile]")
{
   QTemporaryFile tmpFile(QDir::tempPath() + "/MyCsCoreTestXXXXXX.name");
   tmpFile.open();

   QRegularExpression regExp("/MyCsCoreTest[A-Za-z]{6}\\.name$");
   QRegularExpressionMatch match = regExp.match(tmpFile.fileName());

   REQUIRE(match.hasMatch() == true);
}

TEST_CASE("QTemporaryFile create_fname_b", "[qtemporaryfile]")
{
   QTemporaryFile tmpFile(QDir::tempPath() + "/XXXXXX.log");
   tmpFile.open();

   QRegularExpression regExp("/[A-Za-z]{6}\\.log$");
   QRegularExpressionMatch match = regExp.match(tmpFile.fileName());

   REQUIRE(match.hasMatch() == true);
}

TEST_CASE("QTemporaryFile create_fname_c", "[qtemporaryfile]")
{
   QTemporaryFile tmpFile(QDir::tempPath() + "/tmpXXXXXX");
   tmpFile.open();

   QRegularExpression regExp("/tmp[A-Za-z]{6}$");
   QRegularExpressionMatch match = regExp.match(tmpFile.fileName());

   REQUIRE(match.hasMatch() == true);
}

TEST_CASE("QTemporaryFile create_fname_d", "[qtemporaryfile]")
{
   QTemporaryFile tmpFile(QDir::tempPath() + "/XXXXXX");
   tmpFile.open();

   QRegularExpression regExp("/[A-Za-z]{6}$");
   QRegularExpressionMatch match = regExp.match(tmpFile.fileName());

   REQUIRE(match.hasMatch() == true);
}

TEST_CASE("QTemporaryFile file_template", "[qtemporaryfile]")
{
   QTemporaryFile tmpFile;
   bool ok = tmpFile.open();

   REQUIRE(ok == true);

   QString result = tmpFile.fileTemplate();

   REQUIRE(result.endsWith("cs_temp.XXXXXX") == true);
   REQUIRE(tmpFile.autoRemove() == true);
}

