/***********************************************************************
*
* Copyright (c) 2012-2026 Barbara Geller
* Copyright (c) 2012-2026 Ansel Sermersheim
*
* Copyright (c) 2015 The Qt Company Ltd.
*
* This file is part of KitchenSink.
*
* KitchenSink is free software which is released under the BSD 2-Clause license.
* For license details refer to the LICENSE provided with this project.
*
* KitchenSink is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
*
* https://opensource.org/licenses/BSD-2-Clause
*
***********************************************************************/

#include "connection.h"
#include "util.h"

#include <QHostInfo>

static constexpr const int TransferTimeout = 30 * 1000;
static constexpr const int PongTimeout     = 60 * 1000;
static constexpr const int PingInterval    = 5 * 1000;
static constexpr const char SeparatorToken = ' ';

Connection::Connection(QObject *parent)
   : QTcpSocket(parent)
{
   greetingMessage = tr("undefined");
   username        = tr("unknown");

   state = WaitingForGreeting;
   currentDataType = Undefined;
   numBytesForCurrentDataType = -1;
   transferTimerId = 0;
   isGreetingMessageSent = false;
   pingTimer.setInterval(PingInterval);

   QObject::connect(this,       &Connection::readyRead,    this,       &Connection::processReadyRead);
   QObject::connect(this,       &Connection::disconnected, &pingTimer, &QTimer::stop);
   QObject::connect(&pingTimer, &QTimer::timeout,          this,       &Connection::sendPing);
   QObject::connect(this,       &Connection::connected,    this,       &Connection::sendGreetingMessage);
}

QString Connection::name() const
{
   return username;
}

void Connection::setGreetingMessage(const QString &message)
{
   greetingMessage = message;
}

bool Connection::sendMessage(const QString &message)
{
   if (message.isEmpty()) {
      return false;
   }

   QByteArray msg = message.toUtf8();
   QByteArray data = "MESSAGE " + QByteArray::number(msg.size()) + ' ' + msg;

   return write(data) == data.size();
}

void Connection::timerEvent(QTimerEvent *timerEvent)
{
   if (timerEvent->timerId() == transferTimerId) {
      abort();
      killTimer(transferTimerId);
      transferTimerId = 0;
   }
}

void Connection::processReadyRead()
{
   if (state == WaitingForGreeting) {
      if (! readProtocolHeader()) {
         return;
      }

      if (currentDataType != Greeting) {
         abort();
         return;
      }

      state = ReadingGreeting;
   }

   if (state == ReadingGreeting) {
      if (! hasEnoughData()) {
         return;
      }

      buffer = read(numBytesForCurrentDataType);

      if (buffer.size() != numBytesForCurrentDataType) {
         abort();
         return;
      }

      QString peerName = peerAddress().toString();

      // try to look up the name of the peer
      QHostInfo peerInfo = QHostInfo::fromName(peerName);

      if (! peerInfo.hostName().isEmpty()) {
         peerName = peerInfo.hostName();
      }

      username = QString(buffer) + '@' + peerName + "  Port: " + QString::number(peerPort());

      currentDataType = Undefined;
      numBytesForCurrentDataType = 0;
      buffer.clear();

      if (! isValid()) {
         abort();
         return;
      }

      if (! isGreetingMessageSent) {
         sendGreetingMessage();
      }

      pingTimer.start();
      pongTime.start();
      state = ReadyForUse;

      emit readyForUse();
   }

   do {
      if (currentDataType == Undefined) {
         if (! readProtocolHeader()) {
            return;
         }
      }

      if (! hasEnoughData()) {
         return;
      }

      processData();

   } while (bytesAvailable() > 0);
}

void Connection::sendPing()
{
   if (pongTime.elapsed() > PongTimeout) {
      abort();
      return;
   }

   write("PING 1 p");
}

void Connection::sendGreetingMessage()
{
   QByteArray greeting = greetingMessage.toUtf8();
   QByteArray data = "GREETING " + QByteArray::number(greeting.size()) + ' ' + greeting;

   if (write(data) == data.size()) {
      isGreetingMessageSent = true;
   }
}

int Connection::readDataIntoBuffer(int maxSize)
{
   if (maxSize > MaxBufferSize) {
      return 0;
   }

   int numBytesBeforeRead = buffer.size();

   if (numBytesBeforeRead == MaxBufferSize) {
      abort();
      return 0;
   }

   while (bytesAvailable() > 0 && buffer.size() < maxSize) {
      buffer.append(read(1));

      if (buffer.endsWith(SeparatorToken)) {
         break;
      }
   }

   return buffer.size() - numBytesBeforeRead;
}

int Connection::dataLengthForCurrentDataType()
{
   if (bytesAvailable() <= 0 || readDataIntoBuffer() <= 0 || !buffer.endsWith(SeparatorToken)) {
      return 0;
   }

   buffer.chop(1);
   int number = buffer.toInt();
   buffer.clear();
   return number;
}

bool Connection::readProtocolHeader()
{
   if (transferTimerId) {
      killTimer(transferTimerId);
      transferTimerId = 0;
   }

   if (readDataIntoBuffer() <= 0) {
      transferTimerId = startTimer(TransferTimeout);
      return false;
   }

   if (buffer == "PING ") {
      currentDataType = Ping;
   } else if (buffer == "PONG ") {
      currentDataType = Pong;
   } else if (buffer == "MESSAGE ") {
      currentDataType = PlainText;
   } else if (buffer == "GREETING ") {
      currentDataType = Greeting;
   } else {
      currentDataType = Undefined;
      abort();
      return false;
   }

   buffer.clear();
   numBytesForCurrentDataType = dataLengthForCurrentDataType();
   return true;
}

bool Connection::hasEnoughData()
{
   if (transferTimerId) {
      QObject::killTimer(transferTimerId);
      transferTimerId = 0;
   }

   if (numBytesForCurrentDataType <= 0) {
      numBytesForCurrentDataType = dataLengthForCurrentDataType();
   }

   if (bytesAvailable() < numBytesForCurrentDataType
         || numBytesForCurrentDataType <= 0) {
      transferTimerId = startTimer(TransferTimeout);
      return false;
   }

   return true;
}

void Connection::processData()
{
   buffer = read(numBytesForCurrentDataType);

   if (buffer.size() != numBytesForCurrentDataType) {
      abort();
      return;
   }

   switch (currentDataType) {
      case PlainText:
         emit newMessage(username, QString::fromUtf8(buffer));
         break;

      case Ping:
         write("PONG 1 p");
         break;

      case Pong:
         pongTime.restart();
         break;

      default:
         break;
   }

   currentDataType = Undefined;
   numBytesForCurrentDataType = 0;
   buffer.clear();
}
